package ru.yandex.partner.core.filter.db;

import java.net.IDN;
import java.util.Collection;
import java.util.function.BinaryOperator;
import java.util.stream.Collectors;

import org.jooq.Condition;

import ru.yandex.partner.core.entity.domain.filter.DomainFilters;
import ru.yandex.partner.core.entity.domain.model.BaseDomain;
import ru.yandex.partner.core.entity.mirror.filter.MirrorFilters;
import ru.yandex.partner.core.entity.mirror.model.BaseMirror;
import ru.yandex.partner.core.filter.CoreFilterNode;
import ru.yandex.partner.core.filter.meta.MetaFilter;
import ru.yandex.partner.core.filter.operator.FilterOperator;
import ru.yandex.partner.core.multistate.mirror.MirrorStateFlag;

import static java.util.function.Predicate.not;
import static ru.yandex.partner.libs.multistate.MultistatePredicates.has;

public class DomainMirrorDbFilter<M> implements DbFilter<M, String> {
    private final MetaFilter<? super M, String> metaFilter;
    private final Class<M> resolvableModelClass;
    private final MatchDbFilter<M, BaseDomain, BaseDomain> domainMatchFilter;
    private final MatchDbFilter<M, BaseMirror, BaseMirror> mirrorMatchFilter;

    public DomainMirrorDbFilter(MetaFilter<? super M, String> metaFilter, Class<M> resolvableModelClass,
                                MatchDbFilter<M, BaseDomain, BaseDomain> domainMatchFilter,
                                MatchDbFilter<M, BaseMirror, BaseMirror> mirrorMatchFilter) {
        this.metaFilter = metaFilter;
        this.resolvableModelClass = resolvableModelClass;
        this.domainMatchFilter = domainMatchFilter;
        this.mirrorMatchFilter = mirrorMatchFilter;
    }

    @Override
    public MetaFilter<? super M, String> getMetaFilter() {
        return metaFilter;
    }

    @Override
    public Class<M> getResolvableModelClass() {
        return resolvableModelClass;
    }

    @Override
    public Condition getCondition(FilterOperator operator, Collection<String> values) {
        Collection<String> domains = toUnicodeDomains(values);

        BinaryOperator<Condition> binaryOperator = switch (operator) {
            case IN, EQUALS, LIKE -> Condition::or;
            default -> Condition::and;
        };


        return binaryOperator.apply(
                domainMatchFilter.getCondition(FilterOperator.MATCH, getDomainFilter(operator, domains)),
                mirrorMatchFilter.getCondition(FilterOperator.MATCH, getMirrorFilter(operator, domains))
        );
    }

    private Collection<String> toUnicodeDomains(Collection<String> domains) {
        return domains.stream()
                .map(IDN::toUnicode)
                .collect(Collectors.toList());
    }

    private CoreFilterNode<BaseDomain> getDomainFilter(FilterOperator operator, Collection<String> domains) {
        return CoreFilterNode.create(DomainFilters.DOMAIN, operator, domains);
    }

    private CoreFilterNode<BaseMirror> getMirrorFilter(FilterOperator operator, Collection<String> domains) {
        return CoreFilterNode.create(MirrorFilters.DOMAIN, operator, domains)
                .and(CoreFilterNode.create(
                                MirrorFilters.MULTISTATE,
                                FilterOperator.IN,
                                not(has(MirrorStateFlag.DELETED)).and(not(has(MirrorStateFlag.REJECTED)))
                        ));
    }
}
