package ru.yandex.partner.core.filter.db;

import java.util.Collection;

import org.jooq.Condition;
import org.jooq.TableField;

import ru.yandex.partner.core.filter.CoreFilterNode;
import ru.yandex.partner.core.filter.container.ModelFilterContainer;
import ru.yandex.partner.core.filter.exceptions.FilterOperatorException;
import ru.yandex.partner.core.filter.meta.MetaFilter;
import ru.yandex.partner.core.filter.operator.FilterOperator;
import ru.yandex.partner.core.filter.utils.FilterUtils;

/**
 *
 * @param <M>
 * @param <V> Тип модели к которой привязан MetaFilter
 * @param <VV> Тип модели который будет определять набор фильтров
 */
public abstract class MatchDbFilter<M, V, VV extends V> implements DbFilter<M, CoreFilterNode<V>> {
    private final MetaFilter<? super M, CoreFilterNode<V>> metaFilter;
    private final Class<M> resolvableModelClass;
    private final Class<VV> matchModelClass;
    private final TableField<?, Object> filterableField;
    private final TableField<?, Object> selectableField;

    public <F> MatchDbFilter(MetaFilter<? super M, CoreFilterNode<V>> metaFilter,
                             Class<M> resolvableModelClass,
                             Class<VV> matchModelClass,
                             TableField<?, F> filterableField,
                             TableField<?, F> selectableField) {
        this.metaFilter = metaFilter;
        this.resolvableModelClass = resolvableModelClass;
        this.matchModelClass = matchModelClass;
        this.filterableField = (TableField<?, Object>) filterableField;
        this.selectableField = (TableField<?, Object>) selectableField;
    }

    protected abstract ModelFilterContainer<? super VV> getModelFilterContainer();

    @Override
    public MetaFilter<? super M, CoreFilterNode<V>> getMetaFilter() {
        return metaFilter;
    }

    @Override
    public Class<M> getResolvableModelClass() {
        return resolvableModelClass;
    }

    @Override
    public Condition getCondition(FilterOperator operator,
                                  Collection<CoreFilterNode<V>> values) {
        var value = FilterUtils.getSingleValueOrThrow(values, operator);
        return switch (operator) {
            case MATCH -> wrapSubCondition(value);
            case NOT_MATCH -> wrapSubConditionNegated(value);
            default -> throw new FilterOperatorException();
        };
    }

    protected Condition wrapSubCondition(CoreFilterNode<? super VV> coreFilterNode) {
        return FilterUtils.wrapSubCondition(filterableField, selectableField,
                ((CoreFilterNode) coreFilterNode).toCondition(matchModelClass, getModelFilterContainer()));
    }

    protected Condition wrapSubConditionNegated(CoreFilterNode<? super VV> coreFilterNode) {
        return FilterUtils.wrapSubConditionNegated(filterableField, selectableField,
                ((CoreFilterNode) coreFilterNode).toCondition(matchModelClass, getModelFilterContainer()));
    }
}
