package ru.yandex.partner.core.filter.db;

import java.util.Collection;
import java.util.List;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import org.jooq.Condition;
import org.jooq.Field;

import ru.yandex.partner.core.filter.meta.MetaFilter;
import ru.yandex.partner.core.filter.operator.FilterOperator;
import ru.yandex.partner.core.filter.utils.FilterUtils;
import ru.yandex.partner.core.multistate.Multistate;
import ru.yandex.partner.core.multistate.StateFlag;
import ru.yandex.partner.libs.multistate.graph.MultistateGraph;

import static ru.yandex.partner.libs.multistate.MultistatePredicates.has;

public class MultistateDbFilter<M, T extends StateFlag<T>> extends BaseDbFilter<M, Long, Object> {
    private final MultistateGraph<M, T> multistateGraph;

    public MultistateDbFilter(MetaFilter<? super M, Object> metaFilter,
                              Class<M> resolvableModelClass,
                              MultistateGraph<M, T> multistateGraph, Field<Long> tableField) {
        super(metaFilter, resolvableModelClass, tableField);
        this.multistateGraph = multistateGraph;
    }

    @Override
    public Condition getCondition(FilterOperator operator, Collection<Object> values) {
        return getConditionImpl(operator, values);
    }

    @Override
    public Condition getCondition(FilterOperator operator, Object value) {
        if (Collection.class.isAssignableFrom(value.getClass())) {
            return getConditionImpl(operator, (Collection<Object>) value);
        }
        return getConditionImpl(operator, List.of(value));
    }

    private Condition getConditionImpl(FilterOperator operator, Collection<Object> values) {
        if (values.isEmpty()) {
            throw new IllegalArgumentException("Collection must be non empty!");
        }
        Object value = values.iterator().next();
        Class<?> valueClass = value.getClass();
        if (Collection.class.isAssignableFrom(valueClass)) {
            return getCondition(operator, value);
        }
        if (Predicate.class.isAssignableFrom(valueClass)) {
            return getConditionFromPredicate(operator, values.stream().map(v -> (Predicate<Multistate<T>>) v).toList());
        }
        if (StateFlag.class.isAssignableFrom(valueClass)) {
            return getConditionFromPredicate(operator, List.of(has((T) value)));
        }
        if (Long.class.isAssignableFrom(valueClass)) {
            return getConditionFromLong(operator, values.stream().map(v -> (Long) v).toList());
        }
        throw new IllegalArgumentException("Type of values should be Long, StateFlag or Predicate!");
    }

    private Condition getConditionFromLong(FilterOperator operator, List<Long> values) {
        Function<Long, Condition> conditionFunction;
        switch (operator) {
            case GREATER:
                conditionFunction = value -> getField().gt(value);
                break;
            case GREATER_OR_EQUALS:
                conditionFunction = value -> getField().ge(value);
                break;
            case LOWER:
                conditionFunction = value -> getField().lt(value);
                break;
            case LOWER_OR_EQUALS:
                conditionFunction = value -> getField().le(value);
                break;
            default:
                return FilterUtils.getConditionInternal(getField(), operator, values);
        }

        Long value = FilterUtils.getSingleValueOrThrow(values, operator);
        return conditionFunction.apply(value);
    }

    private Condition getConditionFromPredicate(FilterOperator operator, List<Predicate<Multistate<T>>> values) {
        var reducedPredicate = values.stream().reduce(Predicate::and)
                .orElseThrow(() -> new IllegalArgumentException("Collection must be non empty!"));

        var multistateValues = multistateGraph.getMultistatesForPredicate(reducedPredicate)
                .stream()
                .map(Multistate::toMultistateValue)
                .distinct()
                // это для sql тестов, иначе флапают
                .sorted()
                .collect(Collectors.toList());

        return getConditionInternal(operator, multistateValues);
    }
}
