package ru.yandex.partner.core.filter.dbmeta;

import java.util.ArrayList;
import java.util.Collection;

import javax.annotation.Nullable;

import org.jooq.Condition;
import org.jooq.impl.DSL;

import ru.yandex.partner.core.filter.FilterType;
import ru.yandex.partner.core.filter.dbmeta.models.PublicIdFilterModel;
import ru.yandex.partner.core.filter.operator.FilterOperator;

public class PublicIdFilter<M> extends BaseFilter<M, PublicIdFilterModel> {
    private final NumberFilter<M, Long> uniqueIdFilter;
    private final NumberFilter<M, Long> blockIdFilter;

    public PublicIdFilter(String name,
                          Class<? super M> modelClass,
                          NumberFilter<M, Long> uniqueIdFilter,
                          NumberFilter<M, Long> blockIdFilter) {
        super(name, FilterType.PUBLIC_ID, modelClass, PublicIdFilterModel.class);
        this.uniqueIdFilter = uniqueIdFilter;
        this.blockIdFilter = blockIdFilter;
    }

    @Override
    public Condition getCondition(FilterOperator operator, Collection<PublicIdFilterModel> values) {
        var blockIds = new ArrayList<Long>(values.size());
        var uniqueIds = new ArrayList<Long>(values.size());

        for (PublicIdFilterModel value : values) {
            switch (value.getSearchType()) {
                case BLOCK_ID:
                    blockIds.add(value.getValue());
                    break;
                case UNIQUE_ID:
                    uniqueIds.add(value.getValue());
                    break;
                case UNKNOWN:
                    // просто пропускаем
                    break;
                default:
                    throw new IllegalArgumentException("Enum value is unsupported. PublicIdSearchType = " + value);
            }
        }


        Condition condition = null;

        if (!blockIds.isEmpty()) {
            condition = blockIdFilter.getCondition(operator, blockIds);
        }

        if (!uniqueIds.isEmpty()) {
            condition = mergeWithOr(condition, uniqueIdFilter.getCondition(operator, uniqueIds));
        }

        return condition != null
                ? condition
                : DSL.falseCondition();
    }

    private Condition mergeWithOr(@Nullable Condition condition1, Condition condition2) {
        if (condition1 == null) {
            return condition2;
        } else {
            return condition1.or(condition2);
        }
    }
}
