package ru.yandex.partner.core.multitype.repository;

import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Sets;
import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.Record;
import org.jooq.SelectQuery;

import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.multitype.entity.JoinCollector;
import ru.yandex.direct.multitype.repository.RepositoryTypeSupportFacade;
import ru.yandex.direct.multitype.typesupport.TypeSupportUtils;

import static com.google.common.base.Preconditions.checkArgument;
import static org.springframework.util.ObjectUtils.isEmpty;
import static ru.yandex.direct.multitype.typesupport.TypeSupportUtils.getObjectsByTypeSupports;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
public abstract class PartnerRepositoryTypeSupportFacade<T extends Model, K, A, U>
        extends RepositoryTypeSupportFacade<T, K, A, U> {
    private final List<? extends PartnerRepositoryTypeSupport<? extends T, A, U>> supports;
    private final Map<? extends PartnerRepositoryTypeSupport<? extends T, A, U>,
            Set<ModelProperty<? extends Model, ?>>> affectedModelPropertiesMap;
    private final Map<? extends PartnerRepositoryTypeSupport<? extends T, A, U>,
            Collection<Field<?>>> allModelFieldsMap;
    private final Map<ModelProperty<? extends Model, ?>, Field<?>> fieldsForModelProperties;
    private final Supplier<T> initializer;
    private final Map<K, Supplier<T>> initializerByType;
    private final Collection<Field<?>> additionalReadFields;

    public PartnerRepositoryTypeSupportFacade(List<? extends PartnerRepositoryTypeSupport<? extends T, A, U>> supports,
                                              Map<K, Supplier<T>> initializerByType,
                                              Set<Field<?>> additionalReadFields,
                                              Set<Class<? extends T>> whiteListSupportTypeClass) {
        this(supports, null, initializerByType, additionalReadFields, whiteListSupportTypeClass);
    }

    public PartnerRepositoryTypeSupportFacade(List<? extends PartnerRepositoryTypeSupport<? extends T, A, U>> supports,
                                              Supplier<T> initializer,
                                              Set<Class<? extends T>> whiteListSupportTypeClass) {
        this(supports, initializer, null, Set.of(), whiteListSupportTypeClass);
    }

    private PartnerRepositoryTypeSupportFacade(List<? extends PartnerRepositoryTypeSupport<? extends T, A, U>> supports,
                                               @Nullable Supplier<T> initializer,
                                               @Nullable Map<K, Supplier<T>> initializerByType,
                                               Set<Field<?>> additionalReadFields,
                                               Set<Class<? extends T>> whiteListSupportTypeClass) {
        // TODO: Add @Nullable annotation in super() constructor
        super(supports, initializerByType, additionalReadFields, whiteListSupportTypeClass);
        this.supports = supports;
        this.initializer = initializer;
        this.initializerByType = initializerByType;

        this.affectedModelPropertiesMap = collectAllAffectedModelProperties();
        this.allModelFieldsMap = collectAllModelFieldsMap();
        this.fieldsForModelProperties = collectFieldsForModelProperties();

        // todo можно из super забрать, если создать метод
        this.additionalReadFields = additionalReadFields;
    }

    /**
     * Проверяем, есть ли в рекорде незаполненные поля
     */
    private static boolean isRecordFilled(Collection<Field<?>> fields, Record record) {
        return StreamEx.of(fields)
                .map(record::getValue)
                .nonNull()
                .findAny()
                .isPresent();
    }

    // TODO: change access in direct to protected/public
    private Supplier<T> getModelInitializer(K type) {
        if (this.initializerByType != null) {
            checkArgument(initializerByType.containsKey(type), "Unsupported type: " + type.toString());
            return initializerByType.get(type);
        } else if (this.initializer != null) {
            return this.initializer;
        }
        throw new IllegalArgumentException("Wrong initializer.");
    }

    private Map<? extends PartnerRepositoryTypeSupport<? extends T, A, U>,
            Set<ModelProperty<? extends Model, ?>>> collectAllAffectedModelProperties() {
        return StreamEx.of(supports)
                .toMap(
                        s -> s,
                        support -> (Set<ModelProperty<? extends Model, ?>>) support.getAffectedModelProperties()
                );
    }

    /**
     * Собрать все поля из базы, относящиеся к модели в виде мапы
     */
    private Map<? extends PartnerRepositoryTypeSupport<? extends T, A, U>, Collection<Field<?>>>
    collectAllModelFieldsMap() {
        return StreamEx.of(supports)
                .toMap(s -> s, PartnerRepositoryTypeSupport::getFields);
    }

    public Set<ModelProperty<? extends Model, ?>> getModelPropertiesByModel(Class<?> clazz) {
        return StreamEx.of(getSupportsByClass(clazz))
                .map(s -> affectedModelPropertiesMap.getOrDefault(s, Collections.emptySet()))
                .flatMap(Collection::stream)
                .collect(Collectors.toUnmodifiableSet());
    }

    /**
     * Просчитать соответствие "свойство модели" => "поле БД" для всех тайпсуппортов.
     * Учитываются только те свойства, которые соответствуют единственному полю.
     * Если какое-то поле по разным тайпсуппортам будет мапиться на разные поля, будет выброшено исключение.
     *
     * @return
     */
    private Map<ModelProperty<? extends Model, ?>, Field<?>> collectFieldsForModelProperties() {
        return affectedModelPropertiesMap.entrySet().stream()
                .flatMap(e -> getOneToOneFieldsForModelProperties(e.getKey(), e.getValue()).entrySet().stream())
                .distinct()
                .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue, (field, field2) -> {
                    throw new IllegalStateException(String.format("Ambiguous mapping of Model property to DB fileds: " +
                            "%s, %s", field, field2));
                }));
    }

    /**
     * Для тайпсуппорта получить соответствие "свойство модели" => "поле БД"
     * Те свойства, которые соответствуют не единственному полю, отбрасываются.
     *
     * @param typeSupport
     * @param value
     * @return
     */
    private Map<? extends ModelProperty<? extends Model, ?>, ? extends Field<?>> getOneToOneFieldsForModelProperties(
            PartnerRepositoryTypeSupport<? extends T, A, U> typeSupport, Set<ModelProperty<? extends Model, ?>> value) {
        return getFieldsForModelProperties(typeSupport, value).entrySet().stream()
                .filter(e1 -> e1.getValue().size() == 1)
                .collect(Collectors.toMap(Map.Entry::getKey, e2 -> e2.getValue().iterator().next()));
    }

    /**
     * Для тайпсуппорта получить соответствие "свойство модели" => "набор полей БД"
     *
     * @param typeSupport
     * @param modelProperties
     * @return
     */
    private Map<ModelProperty<? extends Model, ?>, Set<Field<?>>> getFieldsForModelProperties(
            PartnerRepositoryTypeSupport<? extends T, A, U> typeSupport,
            Set<ModelProperty<? extends Model, ?>> modelProperties) {
        return modelProperties.stream().collect(Collectors.toMap(
                Function.identity(), property -> typeSupport.getFields(Set.of(property)))
        );
    }


    public List<? extends PartnerRepositoryTypeSupport<? extends T, A, U>> getSupportsByModelProperties(
            List<? extends PartnerRepositoryTypeSupport<? extends T, A, U>> supports,
            @Nonnull Set<ModelProperty<? extends Model, ?>> modelProperties) {
        return StreamEx.of(supports)
                .filter(s -> !Sets.intersection(s.getAffectedModelProperties(), modelProperties).isEmpty())
                .toList();
    }

    /**
     * Получить поля из базы, которые требуются для обработки переданнх modelProperties
     */
    public Collection<Field<?>> getSelectedFields(@Nonnull Set<ModelProperty<? extends Model, ?>> modelProperties) {
        // Определяем тайпсаппорты, которые обрабатывают переданные ModelProperty
        List<? extends PartnerRepositoryTypeSupport<? extends T, A, U>> selectedSupports =
                getSupportsByModelProperties(supports, modelProperties);
        // Для отобранных тайпсаппортов получаем список полей из базы
        return StreamEx.of(selectedSupports)
                .map(s -> allModelFieldsMap.getOrDefault(s, Collections.emptySet()))
                .flatMap(Collection::stream)
                .append(additionalReadFields)
                .sorted((Comparator.comparing(Field::getName)))
                .collect(Collectors.toCollection(LinkedHashSet::new));
    }

    public Optional<Field<?>> getFieldForModelProperty(@Nonnull ModelProperty<? extends Model, ?> modelProperty) {
        return Optional.ofNullable(fieldsForModelProperties.get(modelProperty));
    }

    /**
     * Получить экземпляр модели по записи в базе
     */
    @Override
    public T getModelFromRecord(Record record) {
        T model = getModelInitializer(getModelType(record)).get();
        List<? extends PartnerRepositoryTypeSupport<? extends T, A, U>> relevantSupports =
                getSupportsByClass(model.getClass());
        fillModelFromRecord(record, model, relevantSupports);
        return model;
    }

    /**
     * Получить экземпляр модели по записи в базе
     * Заполняются только поля необходимые в указанных классах
     */
    @Override
    public T getModelFromRecord(Record record, Collection<Class<? extends T>> classes) {
        T model = getModelInitializer(getModelType(record)).get();
        Class<? extends Model> modelClass = model.getClass();
        classes.forEach(clazz -> checkArgument(clazz.isAssignableFrom(modelClass),
                "Model %s doesn't implement/extend requested %s", modelClass, clazz));

        List<? extends PartnerRepositoryTypeSupport<? extends T, A, U>> relevantSupports =
                TypeSupportUtils.getSupportsByClasses(supports, classes);
        fillModelFromRecord(record, model, relevantSupports);
        return model;
    }

    /**
     * Получить экземпляр модели по записи в базе
     * Заполняются только поля необходимые для переданны mode properties
     */
    public <M extends T> M getModelFromRecord(Record record,
                                              @Nullable Set<ModelProperty<? extends Model, ?>> modelProperties) {
        T model = getModelInitializer(getModelType(record)).get();
        List<? extends PartnerRepositoryTypeSupport<? extends T, A, U>> relevantSupports =
                getSupportsByClass(model.getClass());
        fillModelFromRecord(
                record, model,
                isEmpty(modelProperties)
                        ? relevantSupports
                        : getSupportsByModelProperties(relevantSupports, modelProperties)
        );
        return (M) model;
    }

    public List<? extends PartnerRepositoryTypeSupport<? extends T, A, U>> getSupportsByClass(Class<?> clazz) {
        return TypeSupportUtils.getSupportsByClass(supports, clazz);
    }

    // TODO: change access in direct to protected/public
    private void fillModelFromRecord(Record record,
                                     T model,
                                     List<? extends PartnerRepositoryTypeSupport<? extends T, A, U>> relevantSupports) {
        relevantSupports.forEach(support -> {
            if (isRecordFilled(support.getFields(), record)) {
                ((PartnerRepositoryTypeSupport) support).fillFromRecord(model, record);
            }
        });
    }

    public void enrichModelFromOtherTables(DSLContext dslContext, List<T> models,
                                           @Nullable Set<ModelProperty<? extends Model, ?>> modelProperties) {
        enrichModelFromOtherTables(dslContext, models,
                isEmpty(modelProperties) ? supports : getSupportsByModelProperties(supports, modelProperties));
    }

    // TODO: make public or protected in direct
    private void enrichModelFromOtherTables(
            DSLContext dslContext,
            List<T> models,
            List<? extends PartnerRepositoryTypeSupport<? extends T, A, U>> relevantSupports) {
        getObjectsByTypeSupports(relevantSupports, models, T::getClass)
                .forEach((support, list) -> enrichModelFromOtherTables(support, dslContext, list));
    }

    // TODO: make public or protected in direct
    private <M extends T> void enrichModelFromOtherTables(PartnerRepositoryTypeSupport<M, A, U> support,
                                                          DSLContext dslContext,
                                                          List<T> models) {
        List<M> typedModels = mapList(models, support.getTypeClass()::cast);
        if (!typedModels.isEmpty()) {
            support.enrichModelFromOtherTables(dslContext, typedModels);
        }
    }

    public <R extends Record> SelectQuery<R> collectSelectJoinStep(
            SelectQuery<R> query,
            @Nullable Set<ModelProperty<? extends Model, ?>> modelProperties) {
        List<? extends PartnerRepositoryTypeSupport<? extends T, A, U>> relevantSupports =
                isEmpty(modelProperties) ? supports : getSupportsByModelProperties(supports, modelProperties);
        return collectSelectJoinStep(query, relevantSupports);
    }

    // TODO: make public or protected in direct
    private <R extends Record> SelectQuery<R> collectSelectJoinStep(
            SelectQuery<R> query, List<? extends PartnerRepositoryTypeSupport<? extends T, A, U>> relevantSupports) {
        var collector = new JoinCollector();
        StreamEx.of(relevantSupports)
                .flatMap(support -> support.joinQuery().stream())
                .forEach(collector::collect);
        return collector.addSelectJoins(query);
    }

    // TODO: change access to public in direct
    protected Predicate<Class<? extends T>> filterClassIsAssignableFromRecord(Record record) {
        Class<? extends Model> modelClass = getModelInitializer(getModelType(record)).get().getClass();
        return clazz -> clazz.isAssignableFrom(modelClass);
    }

    public Set<ModelProperty<? extends Model, ?>>
    extendByAffectedProperties(Set<ModelProperty<? extends Model, ?>> requestedProperties) {
        return getSupportsByModelProperties(supports, requestedProperties)
                .stream()
                .flatMap(support -> support.getAffectedModelProperties().stream())
                .map(it -> (ModelProperty<? extends Model, ?>) it)
                .collect(Collectors.toSet());
    }

    protected void addModelConditionsToQuery(SelectQuery<?> query, @Nullable Class<? extends T> clazz) {
    }

}
