package ru.yandex.partner.core.multitype.repository;

import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.Record;
import org.jooq.Record1;
import org.jooq.Result;
import org.jooq.SelectQuery;

import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.model.ModelWithId;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.multitype.repository.TypedRepository;
import ru.yandex.direct.multitype.repository.container.RepositoryContainer;
import ru.yandex.direct.multitype.repository.filter.Filter;
import ru.yandex.partner.core.configuration.DslContextProviderStub;
import ru.yandex.partner.core.entity.Query;
import ru.yandex.partner.core.entity.QueryOpts;
import ru.yandex.partner.core.filter.CoreFilterNode;
import ru.yandex.partner.core.filter.container.ModelFilterContainer;
import ru.yandex.partner.core.utils.OrderBy;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static org.springframework.util.ObjectUtils.isEmpty;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.partner.core.utils.FunctionalUtils.mapList;

public abstract class PartnerTypedRepository<T extends ModelWithId, K,
        A extends RepositoryContainer, U extends RepositoryContainer> extends TypedRepository<T, K, A, U> {

    protected final DSLContext dslContext;

    private final PartnerRepositoryTypeSupportFacade<T, K, A, U> typeSupportFacade;

    private final Collection<Field<?>> allFields;

    private final ModelFilterContainer<T> modelFilterContainer;

    public PartnerTypedRepository(DSLContext dslContext,
                                  PartnerRepositoryTypeSupportFacade<T, K, A, U> typeSupportFacade,
                                  ModelFilterContainer<T> modelFilterContainer) {
        super(new DslContextProviderStub(dslContext), typeSupportFacade);
        this.typeSupportFacade = typeSupportFacade;
        this.dslContext = dslContext;

        this.allFields = typeSupportFacade.getAllModelFields();

        this.modelFilterContainer = modelFilterContainer;
    }

    public <B extends T> List<T> getAll(Query<B> queryOpts) {
        QueryOpts<B> opts = (QueryOpts<B>) queryOpts; // единственная реализация Query

        Set<ModelProperty<? extends Model, ?>> allRequiredProperties =
                collectAllRequiredProperties(getCompletedModelProperties(opts.getProps()), opts.getOrderByList());

        // сюда передаём поля, которые нужно выбрать из базы, а так же те, по которым сортируем,
        // чтобы при необходимости приджойнить нужные таблицы

        SelectQuery<Record> query = getBaseQuery(this.dslContext, allRequiredProperties, opts.getClazz());
        CoreFilterNode<? super B> coreFilterNode = opts.getFilter();
        if (opts.hasIds()) {
            coreFilterNode = CoreFilterNode.and(coreFilterNode, getCoreFilterNodeById(opts.getIds()));
        }
        applyFilterToQuery(opts.getClazz(), coreFilterNode, query);
        applyOrderToQuery(opts.getOrderByList(), query);

        // сюда передаём только те поля, которые на самом деле нужно выбрать из базы
        return getTyped(
                this.dslContext,
                query,
                null,
                opts.getLimitOffset(),
                allRequiredProperties,
                opts.getClazz(),
                opts.isForUpdate(),
                opts.isBatchOpts()
        );
    }

    public List<T> getTyped(Collection<Long> modelIds, boolean forUpdate) {
        return getTyped(this.dslContext, modelIds, forUpdate);
    }

    private List<T> getTyped(DSLContext dslContext, Collection<Long> modelIds, boolean forUpdate) {
        return getTyped(dslContext, modelIds, Collections.emptySet(), forUpdate);
    }

    private List<T> getTyped(DSLContext dslContext, Collection<Long> modelIds,
                             Set<ModelProperty<? extends Model, ?>> modelProperties, boolean forUpdate) {
        return getTypedModelByIds(dslContext, getIdFilter(modelIds), modelProperties, forUpdate);
    }

    public <M extends T> List<M> getSafely(
            Collection<Long> modelIds,
            Class<M> clazz) {
        List<M> models = getSafely(this.dslContext, modelIds, clazz);
        return models;
    }

    @Override
    public List<T> getSafely(
            DSLContext dslContext,
            Filter filter,
            Collection<Class<? extends T>> classes) {
        if (filter.isEmpty()) {
            return emptyList();
        }

        SelectQuery<Record> query = selectClassFields(dslContext, classes);
        typeSupportFacade.collectSelectJoinStep(query, classes);

        filter.apply(query);

        Result<Record> records = query.fetch();
        List<T> models =
                StreamEx.of(records)
                        .mapToEntry(Function.identity(), r -> filterList(classes,
                                typeSupportFacade.filterClassIsAssignableFromRecord(r)))
                        .removeValues(List::isEmpty)
                        .mapKeyValue(typeSupportFacade::getModelFromRecord)
                        .toList();
        typeSupportFacade.enrichModelFromOtherTables(dslContext, models, classes);
        return models;
    }

    public <M extends T> List<M> getStrictly(
            Collection<Long> modelIds,
            Class<M> clazz) {
        return getStrictly(this.dslContext, modelIds, clazz);
    }

    public <M extends T> List<M> getStrictly(
            DSLContext dslContext,
            Collection<Long> modelIds,
            Class<M> clazz) {
        if (modelIds.isEmpty()) {
            return emptyList();
        }

        SelectQuery<Record> query = selectClassFields(dslContext, singletonList(clazz));
        typeSupportFacade.collectSelectJoinStep(query, clazz);
        getIdFilter(modelIds).apply(query);

        Result<Record> records = query.fetch();
        List<T> models = mapList(records, rec -> typeSupportFacade.getModelFromRecord(rec, clazz));
        typeSupportFacade.enrichModelFromOtherTables(dslContext, models, clazz);

        List<M> filteredModels = StreamEx.of(models)
                .select(clazz)
                .toList();
        checkState(models.size() == filteredModels.size(),
                "some models selected by ids %s is not extend class %s", modelIds, clazz.getName());
        return filteredModels;
    }

    public <M extends T> List<M> getStrictlyFullyFilled(Collection<Long> modelIds, Class<M> clazz, boolean forUpdate) {
        return getStrictlyFullyFilled(this.dslContext, modelIds, clazz, forUpdate);
    }

    public <M extends T> List<M> getStrictlyFullyFilled(DSLContext dslContext,
                                                        Collection<Long> modelIds,
                                                        Class<M> clazz, boolean forUpdate) {
        return getStrictlyFullyFilled(dslContext, modelIds, Collections.emptySet(), clazz, forUpdate);
    }

    public <M extends T> List<M> getStrictlyFullyFilled(Collection<Long> modelIds,
                                                        Set<ModelProperty<? extends Model, ?>> modelProperties,
                                                        Class<M> clazz, boolean forUpdate) {
        return getStrictlyFullyFilled(this.dslContext, modelIds, modelProperties, clazz, forUpdate);
    }

    public <M extends T> List<M> getStrictlyFullyFilled(DSLContext dslContext,
                                                        Collection<Long> modelIds,
                                                        Set<ModelProperty<? extends Model, ?>> modelProperties,
                                                        Class<M> clazz, boolean forUpdate) {
        List<T> models = getTyped(dslContext, modelIds, modelProperties, forUpdate);
        List<M> filteredModels = StreamEx.of(models)
                .select(clazz)
                .toList();
        checkState(models.size() == filteredModels.size(),
                "some models selected by ids %s is not extend class %s", modelIds, clazz.getName());
        return filteredModels;
    }

    protected SelectQuery<Record> getBaseQuery(DSLContext dslContext,
                                               Set<ModelProperty<? extends Model, ?>> modelProperties,
                                               @Nullable Class<? extends T> clazz) {
        SelectQuery<Record> query = selectDbFields(dslContext, modelProperties);
        typeSupportFacade.addModelConditionsToQuery(query, clazz);
        return typeSupportFacade.collectSelectJoinStep(query, modelProperties);
    }

    public List<T> getTypedModelByIds(Filter filter) {
        if (filter.isEmpty()) {
            return emptyList();
        }

        return getTypedModelByIds(dslContext, filter);
    }

    public List<T> getTypedModelByIds(Filter filter,
                                      @Nullable LimitOffset limitOffset
    ) {
        if (filter.isEmpty()) {
            return emptyList();
        }
        // так как здесь идет запрос по айдишникам можно передавать базовый класс
        return getTyped(dslContext, null, filter, limitOffset, null, null, false, false);
    }

    private List<T> getTypedModelByIds(DSLContext dslContext, Filter filter) {
        if (filter.isEmpty()) {
            return emptyList();
        }
        // так как здесь идет запрос по айдишникам можно передавать базовый класс
        return getTyped(dslContext, null, filter, null, null, false);
    }

    private List<T> getTypedModelByIds(DSLContext dslContext, Filter filter,
                                       Set<ModelProperty<? extends Model, ?>> modelProperties,
                                       boolean forUpdate) {
        if (filter.isEmpty()) {
            return emptyList();
        }
        // так как здесь идет запрос по айдишникам можно передавать базовый класс
        return getTyped(dslContext, null, filter, null, modelProperties,
                null, forUpdate, false);
    }

    private SelectQuery<Record> selectDbFields(DSLContext dslContext,
                                               @Nullable Set<ModelProperty<? extends Model, ?>> modelProperties) {
        return dslContext
                .select(isEmpty(modelProperties)
                        ? allFields
                        : typeSupportFacade.getSelectedFields(modelProperties))
                .from(getBaseTable())
                .getQuery();
    }

    protected SelectQuery<Record1<Integer>> getSelectCountQuery(DSLContext dslContext,
                                                                @Nullable Class<? extends T> clazz) {
        var q = dslContext
                .selectCount()
                .from(getBaseTable())
                .getQuery();
        typeSupportFacade.addModelConditionsToQuery(q, clazz);
        return q;
    }

    private SelectQuery<Record> selectClassFields(DSLContext dslContext, Collection<Class<? extends T>> classes) {
        return dslContext
                .select(typeSupportFacade.getModelFields(classes))
                .from(getBaseTable())
                .getQuery();
    }

    public List<T> getTyped(DSLContext dslContext,
                            @Nullable SelectQuery<Record> query,
                            @Nullable Filter filter,
                            @Nullable LimitOffset limitOffset,
                            Class<? extends T> clazz,
                            boolean forUpdate) {

        return getTyped(dslContext, query, filter, limitOffset, null, clazz, forUpdate, false);
    }

    @SuppressWarnings("ParameterNumber")
    protected List<T> getTyped(DSLContext dslContext,
                               @Nullable SelectQuery<Record> query,
                               @Nullable Filter filter,
                               @Nullable LimitOffset limitOffset,
                               @Nullable Set<ModelProperty<? extends Model, ?>> modelProperties,
                               @Nullable Class<? extends T> clazz,
                               boolean forUpdate,
                               boolean isBatch) {
        if (query == null) {
            query = getBaseQuery(dslContext, modelProperties, clazz);
        }

        query.setForUpdate(forUpdate);

        if (filter != null) {
            filter.apply(query);
        }

        if (isBatch && limitOffset != null) {
            query.addLimit(limitOffset.limit()); // ignore offset, where id > lastId
        } else if (limitOffset != null) {
            query.addOffset(limitOffset.offset());
            query.addLimit(limitOffset.limit());
        }
        Result<Record> records = query.fetch();

        List<T> models;
        models = mapList(records, r -> typeSupportFacade.getModelFromRecord(r, modelProperties));
        typeSupportFacade.enrichModelFromOtherTables(dslContext, models, modelProperties);

        return models;
    }

    /**
     * Собрать все свойства, которые нужно запросить.
     * Пустое множество requestedProperties - маркер того, что запросить надо все, в этом случае оставляем его таковыс
     * Иначе добавляем к этому множеству те поля, по которым будем сортировать.
     *
     * @param requestedProperties
     * @param orderByList
     * @return
     */
    protected Set<ModelProperty<? extends Model, ?>> collectAllRequiredProperties(
            @Nullable Set<ModelProperty<? extends Model, ?>> requestedProperties, @Nullable List<OrderBy> orderByList) {
        if (isEmpty(requestedProperties)) {
            return requestedProperties;
        }
        Set<ModelProperty<? extends Model, ?>> allRequiredProperties = new HashSet<>(requestedProperties);
        // extend with support-joins
        allRequiredProperties.addAll(
                typeSupportFacade.extendByAffectedProperties(requestedProperties)
        );
        if (!isEmpty(orderByList)) {
            allRequiredProperties.addAll(
                    orderByList.stream().map(OrderBy::getModelProperty).collect(Collectors.toSet())
            );
        }
        return allRequiredProperties;
    }

    protected <B extends T> void applyFilterToQuery(
            Class<B> searchClass, CoreFilterNode<? super B> coreFilterNode, SelectQuery<?> query) {
        query.addConditions(((CoreFilterNode) coreFilterNode).toCondition(searchClass, modelFilterContainer));
    }

    protected void applyOrderToQuery(@Nullable List<OrderBy> orderByList, SelectQuery<Record> query) {
        if (orderByList != null) {
            for (OrderBy orderBy : orderByList) {
                ModelProperty<? extends Model, ?> modelProperty = orderBy.getModelProperty();
                Optional<Field<?>> field = typeSupportFacade.getFieldForModelProperty(modelProperty);
                field.ifPresent(value -> query.addOrderBy(value.sort(orderBy.getDirection().getSortOrder())));
            }
        }
    }

    protected abstract <S extends T> CoreFilterNode<S> getCoreFilterNodeById(Collection<Long> ids);

    protected abstract ModelProperty<? extends Model, Long> getIdModelProperty();

    protected Set<ModelProperty<? extends Model, ?>> getCompletedModelProperties(
            @Nullable Set<ModelProperty<? extends Model, ?>> modelProperties) {
        // без ID работать не будет
        Set<ModelProperty<? extends Model, ?>> checkedModelProperties;
        if (modelProperties == null || modelProperties.isEmpty()) {
            // если запрашивается пустой список полей, то тягаются все
            // см PartnerRepositoryTypeSupportFacade->collectSelectJoinStep
            // не всегда нужно тягать все поля, это нужно фиксить
            checkedModelProperties = Set.of();
        } else if (!modelProperties.contains(getIdModelProperty())) {
            checkedModelProperties = new HashSet<>(modelProperties);
            checkedModelProperties.add(getIdModelProperty());
        } else {
            checkedModelProperties = modelProperties;
        }
        return checkedModelProperties;
    }

    public <S extends T> Long getEntityCountByCondition(
            CoreFilterNode<? super S> coreFilterNode, Class<S> clazz) {

        SelectQuery<Record1<Integer>> query = getSelectCountQuery(this.dslContext, clazz);
        applyFilterToQuery(clazz, coreFilterNode, query);
        return query.fetchSingleInto(Long.class);
    }
}
