package ru.yandex.partner.core.multitype.repository.relation;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.jooq.DSLContext;

import ru.yandex.partner.core.entity.PageBlockIds;
import ru.yandex.partner.core.entity.common.ModelWithBlockPageIds;
import ru.yandex.partner.core.entity.simplemodels.AbstractSimpleRepository;
import ru.yandex.partner.core.filter.CoreFilterNode;
import ru.yandex.partner.core.filter.meta.MetaFilter;
import ru.yandex.partner.core.filter.operator.BinaryOperator;

public interface RelationRepository<M> {

    static <M extends ModelWithBlockPageIds> RelationRepository<M> of(
            AbstractSimpleRepository<M, ?> simpleRepository,
            MetaFilter<? extends M, Long> pageId,
            MetaFilter<? extends M, Long> blockId
    ) {
        return new RelationRepository<>() {
            @Override
            public Map<PageBlockIds, List<M>> select(DSLContext innerDslContext,
                                                     Collection<PageBlockIds> pageBlockIdsList) {
                if (pageBlockIdsList.isEmpty()) {
                    return Map.of();
                }

                var filter = pageBlockIdsList
                        .stream()
                        .collect(
                                Collectors.groupingBy(PageBlockIds::getPageId,
                                        Collectors.mapping(PageBlockIds::getBlockId, Collectors.toList()))
                        )
                        .entrySet()
                        .stream()
                        .map(entry -> new CoreFilterNode(BinaryOperator.AND, List.of(
                                CoreFilterNode.eq(pageId, entry.getKey()),
                                CoreFilterNode.in(blockId, entry.getValue())
                        ))).collect(Collectors.toList());

                CoreFilterNode<M> filterNode = new CoreFilterNode(BinaryOperator.OR, filter);

                return simpleRepository.getAll(filterNode, null, null, false)
                        .stream().collect(Collectors.groupingBy(r -> new PageBlockIds(r.getPageId(), r.getBlockId())));
            }

            @Override
            public void insert(Collection<M> modelsToInsert) {
                simpleRepository.insert(new ArrayList<>(modelsToInsert));
            }

            @Override
            public void delete(Collection<M> currentModels) {
                simpleRepository.delete(currentModels);
            }
        };


    }

    Map<PageBlockIds, List<M>> select(DSLContext innerDslContext, Collection<PageBlockIds> pageBlockIdsList);

    void insert(Collection<M> modelsToInsert);

    void delete(Collection<M> currentModels);
}
