package ru.yandex.partner.core.multitype.service.validation.type;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.BiPredicate;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import one.util.streamex.EntryStream;

import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.model.ModelWithId;
import ru.yandex.direct.multitype.repository.container.RepositoryContainer;
import ru.yandex.direct.multitype.service.type.update.UpdateOperationContainer;
import ru.yandex.direct.multitype.service.validation.type.ValidationTypeSupportUtils;
import ru.yandex.direct.multitype.typesupport.TypeSupportAffectionHelper;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.partner.core.multitype.repository.PartnerRepositoryTypeSupport;

import static ru.yandex.direct.multitype.service.validation.type.ValidationTypeSupportUtils.buildSubValidationResult;
import static ru.yandex.direct.multitype.service.validation.type.ValidationTypeSupportUtils.filterIndexes;
import static ru.yandex.direct.multitype.service.validation.type.ValidationTypeSupportUtils.filterValidSubResults;
import static ru.yandex.direct.multitype.service.validation.type.ValidationTypeSupportUtils.filterValues;
import static ru.yandex.direct.multitype.service.validation.type.ValidationTypeSupportUtils.select;
import static ru.yandex.direct.multitype.service.validation.type.ValidationTypeSupportUtils.selectAppliedChanges;


public class ValidationTypeSupportFacade<T extends ModelWithId, UC extends UpdateOperationContainer<?>,
        RC extends RepositoryContainer> {

    private final List<? extends ValidationTypeSupport<? extends T, UC>> supports;
    private final List<? extends PartnerRepositoryTypeSupport<? extends T, RC, RC>> repositoryTypeSupports;
    private final TypeSupportAffectionHelper<T> typeSupportAffectionHelper;
    private final Supplier<UC> containerCreator;

    public ValidationTypeSupportFacade(
            List<? extends ValidationTypeSupport<? extends T, UC>> supports,
            List<? extends PartnerRepositoryTypeSupport<? extends T, RC, RC>> repositoryTypeSupports,
            TypeSupportAffectionHelper<T> typeSupportAffectionHelper,
            Supplier<UC> containerCreator) {
        this.supports = supports;
        this.repositoryTypeSupports = repositoryTypeSupports;
        this.typeSupportAffectionHelper = typeSupportAffectionHelper;
        this.containerCreator = containerCreator;
    }

    public Set<ModelProperty<?, ?>> getPropertiesForValidate(Set<ModelProperty<?, ?>> updateProperties,
                                                             Class<? extends T> tClass) {

        // fake model changes
        var modelChanges = new ModelChanges<>(0L, tClass);
        for (ModelProperty updateProperty : updateProperties) {
            modelChanges.process(null, updateProperty);
        }

        var container = containerCreator.get();
        var affectedValidationTypeSupport = supports.stream()
                .filter(support -> typeSupportAffectionHelper.isModelChangesAffectsSupport(
                        container,
                        modelChanges,
                        support.getTypeClass()
                        )
                )
                .collect(Collectors.toList());


        var affectedRepositoryTypeSupports = repositoryTypeSupports.stream()
                .filter(repositorySupport -> affectedValidationTypeSupport.stream().anyMatch(validationSupport ->
                        repositorySupport.getTypeClass().isAssignableFrom(validationSupport.getTypeClass())))
                .collect(Collectors.toSet());

        return affectedRepositoryTypeSupports.stream().map(PartnerRepositoryTypeSupport::getAffectedModelProperties)
                .flatMap(Set::stream)
                .collect(Collectors.toSet());
    }

    public void addPreValidate(UC container, ValidationResult<? extends List<? extends T>, Defect> vr) {
        supports.forEach(support -> addPreValidateBySupport(support, container, vr));
    }

    private <M extends T> void addPreValidateBySupport(
            ValidationTypeSupport<M, UC> support,
            UC container,
            ValidationResult<? extends List<? extends T>, Defect> vr) {
        var subVr = select(vr, support.getTypeClass());

        if (!subVr.getValue().isEmpty()) {
            support.addPreValidate(container, subVr);
        }
    }

    public void updateValidateModelChanges(
            UC container,
            ValidationResult<? extends List<? extends ModelChanges<? extends T>>, Defect> vr) {

        affectedSupports(vr).forEach(support -> updateValidateModelChangesBySupport(container, support, vr));
    }

    private <T2 extends T> void updateValidateModelChangesBySupport(
            UC container, ValidationTypeSupport<T2, UC> support,
            ValidationResult<? extends List<? extends ModelChanges<? extends T>>, Defect> vr) {
        var subVr = ValidationTypeSupportUtils.selectModelChanges(vr, container,
                support.getTypeClass(), typeSupportAffectionHelper);

        if (!subVr.getValue().isEmpty()) {
            support.updateValidateModelChanges(container, subVr);
        }
    }

    public <T2 extends T> void updateValidateBeforeApply(
            UC container,
            ValidationResult<List<ModelChanges<T2>>, Defect> vr,
            Map<Long, ? extends T> unmodifiedModels) {
        ValidationResult<List<ModelChanges<T2>>, Defect> vrWithValidSubResults = filterValidSubResults(vr);
        affectedSupports(vr).forEach(support ->
                updateValidateBeforeApplyBySupport(container, support, vrWithValidSubResults, unmodifiedModels));
    }

    private <T2 extends T, T3 extends T> void updateValidateBeforeApplyBySupport(
            UC container,
            ValidationTypeSupport<T2, UC> support,
            ValidationResult<List<ModelChanges<T3>>, Defect> vr,
            Map<Long, ? extends T> unmodifiedModels
    ) {
        var subVr = ValidationTypeSupportUtils.selectModelChanges(vr, container,
                support.getTypeClass(), typeSupportAffectionHelper);

        if (!subVr.getValue().isEmpty()) {
            Map<Long, T2> typedUnmodifiedModels = EntryStream.of(unmodifiedModels)
                    .selectValues(support.getTypeClass())
                    .toMap();

            support.updateValidateBeforeApply(container, subVr, typedUnmodifiedModels);
        }
    }

    public void updateValidateAppliedChanges(
            UC container,
            ValidationResult<? extends List<? extends T>, Defect> vr,
            Map<Integer, ? extends AppliedChanges<? extends T>> appliedChangesForValidModelChanges) {
        affectedSupports(appliedChangesForValidModelChanges).forEach(support -> {
            var applicableAppliedChanges = typeSupportAffectionHelper
                    .selectAppliedChangesThatAffectSupport(appliedChangesForValidModelChanges,
                            support.getTypeClass());
            if (applicableAppliedChanges.size() > 0) {
                updateValidateAppliedChanges(container, support, vr, applicableAppliedChanges);
            }
        });
    }

    private <T2 extends T> void updateValidateAppliedChanges(
            UC container,
            ValidationTypeSupport<T2, UC> support,
            ValidationResult<? extends List<? extends T>, Defect> vr,
            Map<Integer, ? extends AppliedChanges<? extends T>> appliedChangesForValidModelChanges) {
        BiPredicate<Integer, T> isValidModel =
                (index, x) -> appliedChangesForValidModelChanges.containsKey(index);
        List<Integer> indexes = filterIndexes(vr, support.getTypeClass(), isValidModel);
        List<T2> values = filterValues(vr, support.getTypeClass(), isValidModel);
        ValidationResult<List<T2>, Defect> subVr = buildSubValidationResult(vr, indexes, values);
        Map<Integer, AppliedChanges<T2>> subAppliedChanges = selectAppliedChanges(indexes,
                appliedChangesForValidModelChanges);
        if (!subVr.getValue().isEmpty()) {
            support.updateValidateAppliedChanges(container, subVr, subAppliedChanges);
        }
    }


    /**
     * Общая валидация моделей для всех операций
     * Метод валидирует всю модель
     *
     * @param container - контейнер с общими данными
     * @param vr        - результат валидации моделей
     */
    public void validate(UC container, ValidationResult<? extends List<? extends T>, Defect> vr) {
        supports.forEach(sup -> fillContainerBySupport(sup, container, vr));
        supports.forEach(sup -> validateBySupport(sup, container, vr));
    }

    /**
     * Общая валидация моделей для всех операций
     * Метод валидирует только "задетую" часть модели
     * NOTE! При валидации пачки моделей,
     * будет валидироваться сумма всех задетых полей, даже если у какой то модели это поле не задето
     *
     * @param container                          - контейнер с общими данными
     * @param vr                                 - результат валидации моделей
     * @param appliedChangesForValidModelChanges - подтвержденные изменения
     */
    public void validate(
            UC container,
            ValidationResult<? extends List<? extends T>, Defect> vr,
            Map<Integer, ? extends AppliedChanges<? extends T>> appliedChangesForValidModelChanges) {
        var affectedSupports = affectedSupports(appliedChangesForValidModelChanges).toList();
        affectedSupports.forEach(support -> fillContainerBySupport(support, container, vr));
        affectedSupports.forEach(support -> validateBySupport(support, container, vr));
    }

    private <M extends T> void fillContainerBySupport(ValidationTypeSupport<M, UC> support,
                                                 UC container,
                                                 ValidationResult<? extends List<? extends T>, Defect> vr) {
        // обходим vr вместе с его вложенными результатами и фильтруем их по классу модели
        var vrWithFilteredValues = select(vr, support.getTypeClass());
        if (!vrWithFilteredValues.getValue().isEmpty()) {
            // Наливаем контейнер данными, которые потребуются для TypeSupport'а
            support.fillContainer(container, vrWithFilteredValues.getValue());
        }
    }


    private <M extends T> void validateBySupport(ValidationTypeSupport<M, UC> support,
                                                 UC container,
                                                 ValidationResult<? extends List<? extends T>, Defect> vr) {
        // обходим vr вместе с его вложенными результатами и фильтруем их по классу модели
        var vrWithFilteredValues = select(vr, support.getTypeClass());
        if (!vrWithFilteredValues.getValue().isEmpty()) {
            // Вызываем общую валидацию
            support.validate(container, vrWithFilteredValues);
        }
    }

    public void forceFillContainerByValidationSupports(UC container) {
        // Тут может заполнятся больше чем нужно, но кажется это не обойти,
        // потому что весь набор моделей заранее неизвестен.
        // Если будем делать по модельно можно будет сделать ещё один метод,
        // который будет наливать для конкретной модели
        supports.forEach(support -> support.fillContainerFullDictionaries(container));
    }

    private Stream<? extends ValidationTypeSupport<? extends T, UC>> affectedSupports(
            Map<Integer, ? extends AppliedChanges<? extends T>> changesMap) {

        return supports.stream()
                .filter(support ->
                        changesMap.values().stream()
                                .anyMatch(changes -> typeSupportAffectionHelper.isAppliedChangesAffectsSupport(changes,
                                        support.getTypeClass()))
                );
    }

    private Stream<? extends ValidationTypeSupport<? extends T, UC>> affectedSupports(
            ValidationResult<? extends List<? extends ModelChanges<? extends T>>, Defect> vr
    ) {
        var container = containerCreator.get();

        return supports.stream()
                .filter(support -> vr.getValue().stream()
                        .anyMatch(changes -> typeSupportAffectionHelper.isModelChangesAffectsSupport(container,
                                changes,
                                support.getTypeClass())));
    }
}
