package ru.yandex.partner.core.multitype.service.validation.type.update

import com.google.common.collect.Sets
import ru.yandex.direct.model.*
import ru.yandex.direct.validation.builder.ItemValidationBuilder
import ru.yandex.direct.validation.builder.ListItemValidator
import ru.yandex.direct.validation.builder.ListValidationBuilder
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.PathHelper
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.validation.wrapper.ModelChangesValidationBuilder
import ru.yandex.partner.core.entity.ModelCopyMapper
import ru.yandex.partner.core.entity.common.editablefields.EditableFieldsService
import ru.yandex.partner.core.holder.ModelPropertiesHolder
import ru.yandex.partner.core.validation.defects.RightsDefects


open class EditableFieldValidator<B : ModelWithId>(
    private val copyMappers: Map<Class<*>, ModelCopyMapper<*>>,
    private val editableFieldsService: EditableFieldsService<B>) {

    fun <D : Defect<*>> updateValidateBeforeApply(
        vr: ValidationResult<List<ModelChanges<B>>, D>,
        unmodifiedValidModels: Map<Long, B>): ValidationResult<List<ModelChanges<B>>, D> {
        return ListValidationBuilder(vr)
            .checkEachBy(permitUpdateEarlyValidator(unmodifiedValidModels))
            .result
    }

    @Suppress("UNCHECKED_CAST")
    private fun <D : Defect<*>> permitUpdateEarlyValidator(unmodifiedValidModels: Map<Long, B>): ListItemValidator<ModelChanges<B>, D> {
        return ListItemValidator { _: Int, modelChanges: ModelChanges<B> ->
            val model = unmodifiedValidModels[modelChanges.id]!!
            val vb = ModelChangesValidationBuilder.of(modelChanges) as ItemValidationBuilder<ModelChanges<B>, D>
            val mapper = copyMappers[model.javaClass]!! as ModelCopyMapper<Any>
            val modelCopy = mapper.copy(model) as B
            val appliedChanges = modelChanges.applyTo(modelCopy)
            validateUpdatePermittedInternal(vb, model, modelCopy, appliedChanges)
            vb.result
        }
    }

    @Suppress("UNCHECKED_CAST")
    private fun <D : Defect<*>> validateUpdatePermittedInternal(
        vb: ItemValidationBuilder<ModelChanges<B>, D>,
        modelBeforeAppliedChanges: B,
        modelWithAppliedChanges: B,
        changes: AppliedChanges<B>
    ) {
        // получаем все доступные пути, возможность редактирования определяется флагом
        // Вычисляем EditableModelPropertiesHolder на основе старой модели
        val oldEditablePropertiesHolder = editableFieldsService
            .calculateEditableModelPropertiesHolder(modelBeforeAppliedChanges, changes)
        // Вычисляем EditableModelPropertiesHolder на основе новой модели
        // Это делается для вычисления возможности редактирования связанных полей
        val newEditablePropertiesHolder = editableFieldsService
            .calculateEditableModelPropertiesHolder(modelWithAppliedChanges, changes)
        // Получаем Set изменённых путей
        val changedProperties = fromAppliedChanges(changes, newEditablePropertiesHolder).properties

        // Получаем Set путей, для которых разрешено редактирование (flag == true)
        val editPermittedModelProperties = newEditablePropertiesHolder.enableProperties
        val forgottenProperties = Sets.difference(
            oldEditablePropertiesHolder.enableProperties,
            newEditablePropertiesHolder.enableProperties
        )
        val disallowedChanges = Sets.difference(
            changedProperties,
            editPermittedModelProperties
        )
        if (!disallowedChanges.isEmpty()) {
            for (propertyPath: List<ModelProperty<*, *>> in disallowedChanges) {
                val forgottenPath = forgottenProperties.contains(propertyPath)
                val outerProperty = propertyPath[0]
                val newOuterValue = changes.getNewValue(outerProperty as ModelProperty<in B, Any>)
                if (propertyPath.size == 1) {
                    if (forgottenPath && newOuterValue == null) {
                        // allow nullify
                        continue
                    }
                    vb.result
                        .getOrCreateSubValidationResult(PathHelper.field(outerProperty),
                            newOuterValue)
                        .addError(RightsDefects.forbiddenToChange() as D)
                } else {
                    // Get inner ModelProperty value
                    val innerProperty = propertyPath[1]
                    val innerValue = innerProperty.getRaw(newOuterValue as Model)
                    if (forgottenPath && innerValue == null) {
                        // allow nullify forgotten field
                        continue
                    }
                    vb.result
                        .getOrCreateSubValidationResult(PathHelper.field(innerProperty), innerValue)
                        .addError(RightsDefects.forbiddenToChange() as D)
                }
            }
        }
    }

    companion object {
        fun <M : Model> fromAppliedChanges(
            changes: AppliedChanges<M>,
            editablePropertiesHolder: ModelPropertiesHolder
        ): ModelPropertiesHolder {
            // Вычисляем Holder с изменёнными ModelProperty, для работы со сложными полями используем данные
            // из permittedPropertiesHolder
            val changedPropertiesHolder = ModelPropertiesHolder()
            // Проходим по всем изменённым ModelProperty
            for (prop in changes.actuallyChangedProps) {
                @Suppress("UNCHECKED_CAST")
                val property = prop as ModelProperty<in M, Any>
                if (editablePropertiesHolder.containsPath(property)) {
                    // Если это простое ModelProperty - permittedPropertiesHolder содержит путь из одного элемента
                    changedPropertiesHolder.addPath(property)
                } else if (editablePropertiesHolder.containsCompositePath(property)) {
                    // Если это сложное ModelProperty - permittedPropertiesHolder содержит сложные пути
                    // от этого элемента
                    val nestedPaths = editablePropertiesHolder.getNestedPaths(property)
                    val newValue = changes.getNewValue(property) as Model?
                    val oldValue = changes.getOldValue(property) as Model?
                    if (newValue == null || oldValue == null) {
                        changedPropertiesHolder.addCompositePaths(nestedPaths)
                    } else {
                        val innerProperties = nestedPaths
                            .map { it[1] }
                            .toSet()
                        for (innerProperty in innerProperties) {
                            if (innerProperty.getRaw(newValue) != innerProperty.getRaw(oldValue)) {
                                changedPropertiesHolder.addPath(property, innerProperty)
                            }
                        }
                    }
                } else {
                    // Не найдено нигде - запрещено редактировать
                    changedPropertiesHolder.addPath(property)
                }
            }
            return changedPropertiesHolder
        }
    }
}
