package ru.yandex.partner.core.operation;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import org.jooq.DSLContext;

import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.model.ModelWithId;
import ru.yandex.direct.multitype.repository.TypeModifyRepository;
import ru.yandex.direct.multitype.repository.container.RepositoryContainer;
import ru.yandex.direct.multitype.service.type.update.UpdateOperationContainer;
import ru.yandex.direct.multitype.service.type.update.UpdateOperationTypeSupportFacade;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.update.AppliedChangesValidatedStep;
import ru.yandex.direct.operation.update.ChangesAppliedStep;
import ru.yandex.direct.operation.update.ExecutionStep;
import ru.yandex.direct.operation.update.ModelChangesValidatedStep;
import ru.yandex.direct.operation.update.SimpleAbstractUpdateOperation;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.partner.core.multitype.repository.PartnerModifyRepository;
import ru.yandex.partner.core.multitype.repository.PartnerTypedRepository;
import ru.yandex.partner.core.multitype.service.validation.type.ValidationTypeSupportFacade;
import ru.yandex.partner.core.validation.HasValidationContainer;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public class PartnerTypedUpdateOperation<
        B extends ModelWithId,
        T extends B,
        C extends UpdateOperationContainer<? super T> & RepositoryContainer
        >
        extends SimpleAbstractUpdateOperation<T, Long> implements HasValidationContainer<T, C> {

    private final DSLContext dslContext;
    private final Function<Collection<Long>, List<T>> getModelsWithLockFunction;
    private final UpdateOperationTypeSupportFacade<? super T, C, C, C> updateOperationTypeSupport;
    private final TypeModifyRepository<B, B, C, C> modifyRepository;
    private final C container;
    private final ValidationTypeSupportFacade<B, C, C> validationTypeSupport;

    @SuppressWarnings("ParameterNumber")
    public PartnerTypedUpdateOperation(
            Class<T> modelClass,
            Applicability applicability,
            List<ModelChanges<T>> modelChanges,
            Set<ModelProperty<?, ?>> sensitiveProperties,
            DSLContext dslContext,
            PartnerTypedRepository<T, ?, ?, C> typedRepository,
            UpdateOperationTypeSupportFacade<? super T, C, C, C> updateOperationTypeSupport,
            C container,
            PartnerModifyRepository<B, B, C, C> modifyRepository,
            ValidationTypeSupportFacade<B, C, C> validationTypeSupport
    ) {
        this(
                applicability,
                modelChanges,
                (id) -> createStub(modelClass, id),
                sensitiveProperties,
                dslContext,
                (ids) -> typedRepository.getTyped(ids, true),
                updateOperationTypeSupport,
                modifyRepository,
                validationTypeSupport,
                container
        );
    }

    @SuppressWarnings("ParameterNumber")
    public PartnerTypedUpdateOperation(
            Class<T> modelClass,
            Applicability applicability,
            List<ModelChanges<T>> modelChanges,
            Set<ModelProperty<?, ?>> sensitiveProperties,
            DSLContext dslContext,
            Function<Collection<Long>, List<T>> getModelsWithLockFunction,
            UpdateOperationTypeSupportFacade<? super T, C, C, C> updateOperationTypeSupport,
            TypeModifyRepository<B, B, C, C> modifyRepository,
            ValidationTypeSupportFacade<B, C, C> validationTypeSupport, C container
    ) {
        this(
                applicability,
                modelChanges,
                (id) -> createStub(modelClass, id),
                sensitiveProperties,
                dslContext,
                getModelsWithLockFunction,
                updateOperationTypeSupport,
                modifyRepository,
                validationTypeSupport,
                container
        );
    }

    @SuppressWarnings("ParameterNumber")
    PartnerTypedUpdateOperation(
            Applicability applicability,
            List<ModelChanges<T>> modelChanges,
            Function<Long, T> modelStubCreator,
            Set<ModelProperty<?, ?>> sensitiveProperties,
            DSLContext dslContext,
            Function<Collection<Long>, List<T>> getModelsWithLockFunction,
            UpdateOperationTypeSupportFacade<? super T, C, C, C> updateOperationTypeSupport,
            TypeModifyRepository<B, B, C, C> modifyRepository,
            ValidationTypeSupportFacade<B, C, C> validationTypeSupport,
            C container
    ) {
        super(applicability, modelChanges, modelStubCreator, sensitiveProperties);
        this.dslContext = dslContext;
        this.getModelsWithLockFunction = getModelsWithLockFunction;
        this.updateOperationTypeSupport = updateOperationTypeSupport;
        this.modifyRepository = modifyRepository;
        this.validationTypeSupport = validationTypeSupport;
        this.container = container;
    }

    private static <T extends ModelWithId> T createStub(Class<T> modelClass, Long id) {
        T stub = null;
        try {
            stub = modelClass.getConstructor().newInstance();
        } catch (Exception e) {
            throw new RuntimeException("No default constructor found for stub creation", e);
        }
        stub.setId(id);
        return stub;
    }

    @Override
    protected ValidationResult<List<ModelChanges<T>>, Defect> validateModelChanges(List<ModelChanges<T>> modelChanges) {
        ValidationResult<List<ModelChanges<T>>, Defect> vr =
                new ValidationResult<>(modelChanges);

        validationTypeSupport.updateValidateModelChanges(getContainer(), vr);

        return vr;
    }

    @Override
    protected Collection<T> getModels(Collection<Long> ids) {
        return getModelsWithLockFunction.apply(ids);
    }

    @Override
    protected void onModelChangesValidated(ModelChangesValidatedStep<T> modelChangesValidatedStep) {
        Collection<ModelChanges<T>> modelChanges =
                modelChangesValidatedStep.getValidModelChanges();
        updateOperationTypeSupport.onModelChangesValidated(getContainer(), modelChanges);
    }

    @Override
    protected ValidationResult<List<ModelChanges<T>>, Defect> validateModelChangesBeforeApply(
            ValidationResult<List<ModelChanges<T>>, Defect> preValidateResult, Map<Long, T> models) {
        validationTypeSupport.updateValidateBeforeApply(getContainer(), preValidateResult, models);
        return preValidateResult;
    }

    @Override
    protected void onChangesApplied(ChangesAppliedStep<T> changesAppliedStep) {
        List<AppliedChanges<T>> validAppliedChanges =
                new ArrayList<>(changesAppliedStep.getAppliedChangesForValidModelChanges());
        updateOperationTypeSupport.onChangesApplied(getContainer(), validAppliedChanges);
    }

    /**
     * В этом вызове проверяются права пользователя на редактирование отдельных полей.
     * Вынесена в этот метод, так как необходимы предыдущие значения всех полей, для правильного сравнения вложенных
     *
     * @param validationResult                   - результат валидации
     * @param appliedChangesForValidModelChanges - AppliedChanges для валидных моделей
     * @return - обновлённый результат валидации
     */
    @Override
    protected ValidationResult<List<T>, Defect> validateAppliedChanges(ValidationResult<List<T>,
            Defect> validationResult, Map<Integer, AppliedChanges<T>> appliedChangesForValidModelChanges) {
        validationTypeSupport.updateValidateAppliedChanges(getContainer(), validationResult,
                appliedChangesForValidModelChanges);
        if (!validationResult.hasAnyErrors()) {
            validationTypeSupport.validate(getContainer(), validationResult, appliedChangesForValidModelChanges);
        }
        return validationResult;
    }

    @Override
    protected void onAppliedChangesValidated(AppliedChangesValidatedStep<T> appliedChangesValidatedStep) {
        var validAppliedChanges = new ArrayList<>(appliedChangesValidatedStep.getValidAppliedChanges());

        updateOperationTypeSupport.onAppliedChangesValidated(getContainer(), validAppliedChanges);
    }

    @Override
    protected void beforeExecution(ExecutionStep<T> executionStep) {
        var appliedChangesForExecution = new ArrayList<>(executionStep.getAppliedChangesForExecution());

        updateOperationTypeSupport.beforeExecution(appliedChangesForExecution, getContainer());
    }

    @Override
    protected List<Long> execute(List<AppliedChanges<T>> applicableAppliedChanges) {
        updateOperationTypeSupport.addToAdditionalActionsContainer(getContainer(),
                getContainer(),
                applicableAppliedChanges
        );
        processUpdateTask(applicableAppliedChanges);
        return mapList(applicableAppliedChanges, b -> b.getModel().getId());
    }

    private void processUpdateTask(List<AppliedChanges<T>> validAppliedChanges) {
        modifyRepository.update(dslContext, getContainer(), validAppliedChanges);
    }

    @Override
    protected void afterExecution(ExecutionStep<T> executionStep) {
        var appliedChangesForExecution = new ArrayList<>(executionStep.getAppliedChangesForExecution());
        updateOperationTypeSupport.afterExecution(appliedChangesForExecution, getContainer());
    }

    @Override
    public C getContainer() {
        return container;
    }
}
