package ru.yandex.partner.core.service.adfox;


import java.util.List;
import java.util.Map;
import java.util.Optional;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.module.kotlin.KotlinModule;
import com.google.common.annotations.VisibleForTesting;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.stereotype.Service;
import org.springframework.web.reactive.function.BodyInserters;
import org.springframework.web.reactive.function.client.WebClient;

import ru.yandex.partner.defaultconfiguration.rpc.RpcConfigProps;

import static ru.yandex.partner.libs.common.PartnerRequestService.retryWithFixedDelay;
import static ru.yandex.partner.libs.common.PartnerRequestService.serviceUnavailable;

@Service
public class AdfoxGraphqlService implements AdfoxService {

    private static final Logger LOGGER = LoggerFactory.getLogger(AdfoxGraphqlService.class);

    private WebClient webClient;
    private RpcConfigProps rpcConfigProps;
    private String graphqlUrl;
    private final ObjectMapper objectMapper = new ObjectMapper()
            .registerModule(new KotlinModule());

    @VisibleForTesting
    public AdfoxGraphqlService() {
    }

    @Autowired
    public AdfoxGraphqlService(
            WebClient webClient,
            AdfoxGraphqlRpcConfig adfoxGraphqlRpcConfig
    ) {
        this.webClient = webClient;
        this.graphqlUrl = adfoxGraphqlRpcConfig.getUrl();
        this.rpcConfigProps = adfoxGraphqlRpcConfig;
    }

    @Override
    public Optional<GraphqlResponse> deleteBlock(List<Long> placeId) {
        String query = """
                mutation ($placeIds: [Identifier!]!) {
                    InternalMobileMediation {
                        deletePlace(placeIds: $placeIds)
                    }
                }""";

        var req = new GraphqlRequest(
                query,
                Map.of("placeIds", placeId)
        );

        return send(req);
    }

    @Override
    public Optional<GraphqlResponse> updateBlockName(Long placeId, String newName) {
        String query = """
                mutation ($placeId: Int!, $name: String!) {
                    InternalMobileMediation {
                        updatePlaceName(placeId: $placeId, name: $name ) { name }
                    }
                }""";

        var req = new GraphqlRequest(
                query,
                Map.of(
                        "placeId", placeId,
                        "name", newName
                )
        );

        return send(req);
    }

    @VisibleForTesting
    public Optional<GraphqlResponse> send(GraphqlRequest req) {
        LOGGER.debug("Sending %s to adfox".formatted(req));
        var resp = webClient.post()
                .uri(graphqlUrl)
                .header("Content-Type", MediaType.APPLICATION_JSON_VALUE)
                .body(BodyInserters.fromValue(req))
                .exchangeToMono(clientResponse -> clientResponse.toEntity(String.class))
                .transform(retryWithFixedDelay(rpcConfigProps))
                .blockOptional()
                .orElse(serviceUnavailable())
                .getBody();

        LOGGER.debug("Got response from adfox: %s".formatted(resp));
        try {
            return Optional.ofNullable(objectMapper.readValue(resp, GraphqlResponse.class));
        } catch (JsonProcessingException e) {
            LOGGER.error("Can't parse adfox response: ", e);
            return Optional.empty();
        }
    }

    @VisibleForTesting
    public ObjectMapper getObjectMapper() {
        return objectMapper;
    }
}
