package ru.yandex.partner.core.service.dbutil;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.List;

import javax.sql.DataSource;

import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.Record;
import org.jooq.Table;
import org.jooq.tools.Convert;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import static com.google.common.base.Preconditions.checkState;
import static java.lang.String.format;
import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.joining;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
public class DbValuesGenerator {
    private final DSLContext dslContext;
    private final DataSource dataSource;

    @Autowired
    public DbValuesGenerator(DSLContext dslContext, DataSource dataSource) {
        this.dslContext = dslContext;
        this.dataSource = dataSource;
    }

    static <T> T normalizeValue(Object value, Field<T> field) {
        return normalizeValue(value, field.getType());
    }

    static <T> T normalizeValue(Object value, Class<T> type) {
        return Convert.convert(value, type);
    }

//    /**
//     * Generates a list of new values for key.
//     * <p>
//     * Single value generation was removed due to performance concerns, it is very slow when used to mass generate
//     ids.
//     * Mass generation, on the other hand, is optimized for single value use also.
//     */
//    List<Number> generateValues(AutoIncrementKey key, int count) {
//        checkNotNull(key, "key cannot be null");
//        @SuppressWarnings("unchecked")
//        List<Number> ids = (List<Number>) runSimpleGenerateValues(key.getTable(), key.getKeyField(), count);
//        logDbShardsIds(key.getName(), ids, ImmutableMap.of());
//        return ids;
//    }
//
//    private <R extends Record, T> List<T> runSimpleGenerateValues(Table<R> table, Field<T> keyField, int count) {
//        checkArgument(count >= 0, "requested values count to generate must be greater than or equal to zero");
//        if (count < 1) {
//            return new ArrayList<>();
//        }
//        InsertSetMoreStep<R> insertStep = getDslContext().insertInto(table)
//                .set(keyField, DSL.defaultValue(keyField));
//        for (int i = 1; i < count; i++) {
//            insertStep = insertStep.newRecord().set(keyField, DSL.defaultValue(keyField));
//        }
//        return insertStep
//                .returning(keyField)
//                .fetch()
//                .map(r -> r.getValue(keyField));
//    }

    public static List<Long> longValues(Iterable<? extends Number> numbers) {
        return mapList(numbers, Number::longValue);
    }

    /**
     * Generates new values for key that is associated with chainKey
     */
//    public List<Number> generateValues(List<?> chainValues) {
//        List<Number> ids = runGenerateValues(
//                key.getTable(), key.getKeyField(), key.getValueField(), chainValues);
//        return ids;
//    }
    public <R extends Record, T1> List<Number> generateValues(Table<R> table, Field<T1> keyField,
                                                              List<?> chainValues) {
        return generateValues(table, keyField, keyField, chainValues);
    }

    public <R extends Record, T1, T2> List<Number> generateValues(Table<R> table, Field<T1> keyField,
                                                                  Field<T2> chainField, List<?> chainValues) {
        if (chainValues.size() == 1) {
            // некрасивое ветвление во имя оптимизации, может и несущественной
            // большинство операций генерируют id по одному
            T1 value = this.dslContext.insertInto(table, chainField)
                    .values(normalizeValue(chainValues.get(0), chainField))
                    .returning(keyField).fetchOne().getValue(keyField);
            checkState(value instanceof Number, "Shard key is not a number");
            return singletonList((Number) value);
        }

        try {
            return runMultiGeneration(table, chainField, chainValues);
        } catch (SQLException ex) {
            throw new RuntimeException(ex);
        }
    }

    /**
     * Генерация при помощи SQL мультизапроса.
     * Существенная оптимизация (десятки раз на ~500 объектах).
     * <p>
     * Метод не покрыт юнит-тестом, т.к. реализация завязана исключительно на MySQL JDBC Driver,
     * а на момент написания модуль dbutil тестируется с H2.
     * При рефакторинге следует проверять core-ными тестами.
     */
    @SuppressWarnings("squid:S1764")
    // относится к строке с while
    <R extends Record, T2> List<Number> runMultiGeneration(
            Table<R> table,
            Field<T2> chainField, List<?> chainValues) throws SQLException {
        String template = format("insert into %s (%s) values (%%s);select last_insert_id();",
                table.getName(), chainField.getName());
        String multiQuerySql = chainValues.stream()
                .map(v -> normalizeValue(v, chainField))
                .map(v -> format(template, v))
                .collect(joining("", "begin;", "commit;"));

        List<Number> ids = new ArrayList<>(chainValues.size());

        try (Connection conn = this.dataSource.getConnection();
             Statement stmt = conn.createStatement()) {
            stmt.execute(multiQuerySql);
            while (stmt.getMoreResults() || stmt.getMoreResults()) {    // каждый второй запрос возвращает ResultSet
                try (ResultSet rs = stmt.getResultSet()) {
                    checkState(rs != null && rs.next(), "No ID obtained");
                    long id = rs.getLong(1);
                    checkState(id > 0, "Got invalid ID");
                    ids.add(id);
                }
            }
            checkState(ids.size() == chainValues.size(), "Expected %s IDs to be generated, got %s",
                    chainValues.size(), ids.size());
            return ids;
        }
    }
}
