package ru.yandex.partner.core.service.entitymanager;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import org.jetbrains.annotations.NotNull;
import org.springframework.stereotype.Service;

import ru.yandex.direct.model.ModelWithId;
import ru.yandex.partner.core.entity.ModelQueryService;
import ru.yandex.partner.core.entity.block.model.BaseBlock;
import ru.yandex.partner.core.entity.block.model.InternalMobileRtbBlock;
import ru.yandex.partner.core.entity.block.model.InternalRtbBlock;
import ru.yandex.partner.core.entity.block.model.MobileRtbBlock;
import ru.yandex.partner.core.entity.block.model.RtbBlock;
import ru.yandex.partner.core.entity.page.model.ContextPage;
import ru.yandex.partner.core.entity.page.model.InternalContextPage;
import ru.yandex.partner.core.entity.page.model.InternalMobileApp;
import ru.yandex.partner.core.entity.page.model.MobileAppSettings;
import ru.yandex.partner.core.entity.page.model.PageWithOwner;
import ru.yandex.partner.core.modelmeta.ModelMetaCollector;
import ru.yandex.partner.core.modelmeta.ModelMetaHolder;
import ru.yandex.partner.libs.multistate.graph.MultistateGraph;

@Service
public class EntityManager {
    private final Map<String, EntityMeta> entitiesMeta;
    private final Map<Class<? extends BaseBlock>, Class<? extends PageWithOwner>> blockToPage = Map.of(
            RtbBlock.class, ContextPage.class,
            InternalRtbBlock.class, InternalContextPage.class,
            MobileRtbBlock.class, MobileAppSettings.class,
            InternalMobileRtbBlock.class,  InternalMobileApp.class
    );

    // Главное не забывать вешать @ModelMeta на новые сущности...
    private final Set<Class<? extends BaseBlock>> allBlockClasses = new HashSet<>();

    public EntityManager(
            List<ModelQueryService<? extends ModelWithId>> modelQueryServices,
            List<MultistateGraph<? extends ModelWithId, ?>> multistateGraphs
    ) {
        Map<Class<? extends ModelWithId>, ModelQueryService<? extends ModelWithId>> modelQueryServiceByClass =
                modelQueryServices.stream().collect(
                        Collectors.toMap(ModelQueryService::getBaseClass, Function.identity())
                );

        Map<Class<? extends ModelWithId>, MultistateGraph<? extends ModelWithId, ?>> multistateGraphByClass =
                multistateGraphs.stream().collect(
                        Collectors.toMap(MultistateGraph::getModelClass, Function.identity())
                );

        List<ModelMetaHolder> modelMetaHolders = ModelMetaCollector.getModelMetaHolders();

        entitiesMeta = Maps.newHashMapWithExpectedSize(modelMetaHolders.size());
        for (ModelMetaHolder modelMetaHolder : modelMetaHolders) {
            Class<? extends ModelWithId> entityClass = modelMetaHolder.getEntityClass();

            ModelQueryService<? extends ModelWithId> modelQueryService = modelQueryServiceByClass
                    .entrySet().stream()
                    .filter(entry -> entry.getKey().isAssignableFrom(entityClass))
                    .findFirst()
                    .orElseThrow(() -> new IllegalStateException("Can't find a suitable ModelQueryService"))
                    .getValue();

            MultistateGraph<? extends ModelWithId, ?> multistateGraph = multistateGraphByClass
                    .entrySet().stream()
                    .filter(entry -> entry.getKey().isAssignableFrom(entityClass))
                    .findFirst()
                    .orElseThrow(() -> new IllegalStateException("Can't find a suitable MultistateGraph"))
                    .getValue();

            entitiesMeta.put(
                    modelMetaHolder.getName(),
                    new EntityMeta(
                            modelMetaHolder.getName(),
                            modelMetaHolder.getEntityClass(),
                            modelQueryService,
                            multistateGraph
                    )
            );
            if (BaseBlock.class.isAssignableFrom(entityClass)) {
                allBlockClasses.add(entityClass.asSubclass(BaseBlock.class));
            }
        }
    }

    public Set<Class<? extends BaseBlock>> getMissedBlocks(Set<Class<? extends BaseBlock>> toCheck) {
        return Sets.difference(allBlockClasses, Sets.intersection(allBlockClasses, toCheck));
    }

    public ModelQueryService<? extends ModelWithId> getModelQueryService(String name) {
        return getEntityMetaByName(name).getModelQueryService();
    }

    public MultistateGraph<? extends ModelWithId, ?> getMultistateGraph(String name) {
        return getEntityMetaByName(name).getMultistateGraph();
    }

    public Class<? extends ModelWithId> getEntityClass(String name) {
        return getEntityMetaByName(name).getEntityClass();
    }

    /**
     * Пример:
     * Class<? extends PageWithMultistate> pageClass = getPageByModelClass(RtbBlock.class, PageWithMultistate.class)
     * @param blockClass класс блока, для которого нужно определить класс площадки
     * @param castExtendsPageClass класс пейджа, к которому закастить результат
     * @return
     */
    @NotNull
    public <T> Class<? extends T> getPageClassByBlockClass(
            Class<? extends ModelWithId> blockClass,
            Class<T> castExtendsPageClass
    ) {
        var pageClass = blockToPage.get(blockClass);
        if (pageClass == null) {
            throw new IllegalArgumentException("block class %s isn't associated with any pages".formatted(blockClass));
        }
        if (!castExtendsPageClass.isAssignableFrom(pageClass)) {
            throw new IllegalArgumentException(
                    "%s must be super class of %s to cast properly".formatted(
                        castExtendsPageClass, pageClass
                    )
            );
        }
        return (Class<? extends T>) blockToPage.get(blockClass);
    }

    public Collection<EntityMeta> getAllEntityMetas() {
        return entitiesMeta.values();
    }

    private EntityMeta getEntityMetaByName(String name) {
        EntityMeta entityMeta = entitiesMeta.get(name);

        if (Objects.isNull(entityMeta)) {
            throw new IllegalArgumentException(String.format("Entity with name \"%s\" not found", name));
        }

        return entityMeta;
    }

}
