package ru.yandex.partner.core.service.integration.balance;

import java.time.LocalDate;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.function.BiConsumer;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.collect.Maps;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.MessageSource;
import org.springframework.context.MessageSourceAware;
import org.springframework.context.support.DelegatingMessageSource;
import org.springframework.context.support.MessageSourceAccessor;
import org.springframework.stereotype.Service;

import ru.yandex.partner.core.CoreConstants;
import ru.yandex.partner.core.entity.balance.model.ActiveContract;
import ru.yandex.partner.core.entity.balance.model.BalanceBank;
import ru.yandex.partner.core.entity.balance.model.BalanceContract;
import ru.yandex.partner.core.entity.balance.model.BalancePerson;
import ru.yandex.partner.libs.extservice.balance.BalanceException;
import ru.yandex.partner.libs.extservice.balance.BalanceService;
import ru.yandex.partner.libs.extservice.balance.method.partnercontract.BalancePartnerContract;
import ru.yandex.partner.libs.extservice.balance.method.partnercontract.Collateral;
import ru.yandex.partner.libs.extservice.balance.method.partnercontract.Contract;

import static java.util.Comparator.comparing;

@Service
public class BalanceDocumentService implements MessageSourceAware {
    private static final Logger LOGGER = LoggerFactory.getLogger(BalanceDocumentService.class);
    private static final Map<Integer, CollateralType> COLLATERAL_TYPE_MAP = createCollateralTypeMap();
    private static final Set<String> TYPES = Set.of("PARTNERS");

    private final BalanceService balanceService;
    private final Cache<Long, List<BalancePartnerContract>> clientIdCache;
    private MessageSourceAccessor messages = new MessageSourceAccessor(new DelegatingMessageSource());

    @Autowired
    public BalanceDocumentService(BalanceService balanceService) {

        this.balanceService = balanceService;

        this.clientIdCache = CacheBuilder.newBuilder()
                .expireAfterWrite(7, TimeUnit.DAYS) // 7 * 60 * 60 * 24
                .build();
    }

    @Override
    public void setMessageSource(@Nonnull MessageSource messageSource) {
        this.messages = new MessageSourceAccessor(messageSource);

    }

    public List<BalancePartnerContract> getPartnerContracts(String externalId) {
        return balanceService.getPartnerContracts(externalId);
    }

    public List<ActiveContract> getActiveContracts(Collection<Long> clientIds) {
        Map<String, BalanceBank> bankCache = Maps.newHashMapWithExpectedSize(clientIds.size());
        return clientIds.stream()
                .map(clientId -> getActiveContract0(clientId, bankCache))
                .collect(Collectors.toList());
    }

    private ActiveContract getActiveContract0(long clientId, Map<String, BalanceBank> bankCache) {
        List<BalancePartnerContract> liveContracts =
                getPartnerContracts(clientId).stream()
                        .filter(contract -> contract.isLive(LocalDate.now()))
                        .collect(Collectors.toList());

        if (liveContracts.isEmpty()) {
            return null;
        } else if (liveContracts.size() > 1) {
            LOGGER.warn("More than 1 live contracts for client_id = {}, count = {}", clientId, liveContracts.size());
        }

        BalancePartnerContract contract = liveContracts.get(0);
        Optional<Contract> contractOptional = contract.getContract();

        Optional<BalancePerson> balancePersonOptional = contract.getPerson()
                .map(person -> {
                    BalanceBank balanceBank;
                    if (StringUtils.isNotEmpty(person.getBik())) {
                        balanceBank = bankCache.computeIfAbsent(person.getBik(),
                                (bik) -> {
                                    var bank = balanceService.getBankByBik(bik);
                                    return bank == null
                                            ? null
                                            : bank.toBalanceBank();
                                });
                    } else if (StringUtils.isNotEmpty(person.getSwift())) {
                        balanceBank = bankCache.computeIfAbsent(person.getSwift(),
                                (swift) -> {
                                    var bank = balanceService.getBankBySwift(swift);
                                    return bank == null
                                            ? null
                                            : bank.toBalanceBank();
                                });
                    } else {
                        balanceBank = null;
                    }

                    BalancePerson balancePerson = person.toBalancePerson(balanceBank);
                    Integer payTo = contractOptional.map(Contract::getPayTo).orElse(null);
                    if (!Objects.equals(2, payTo)) {
                        // PI-9937
                        balancePerson.withYamoneyWallet(null);
                    }

                    return balancePerson;
                });

        Optional<BalanceContract> balanceContractOptional = contractOptional.map(this::toBalanceContract);

        return new ActiveContract()
                .withPerson(balancePersonOptional.orElse(null))
                .withContract(balanceContractOptional.orElse(null));
    }

    public List<BalancePartnerContract> getPartnerContracts(long clientId) {
        List<BalancePartnerContract> contracts = getRawPartnerContracts(clientId)
                .stream()
                .filter(contract -> contract.getContract().isPresent())
                .filter(contract -> TYPES.contains(contract.getContract().get().getType()))
                .sorted(comparing((BalancePartnerContract bc) ->
                        bc.getContract().map(Contract::getDt).orElse(LocalDate.EPOCH))
                        .reversed()
                )
                .collect(Collectors.toList());

        for (BalancePartnerContract contract : contracts) {
            contract.collateralDtSortedStream()
                    .filter(collateral -> collateral.isLive(LocalDate.now()))
                    .forEach(collateral -> {
                        CollateralType collateralType = COLLATERAL_TYPE_MAP.get(collateral.getCollateralTypeId());
                        if (collateralType != null && contract.getContract().isPresent()) {
                            collateralType.changeFields(contract.getContract().get(), collateral);
                        }
                    });
        }

        return contracts;
    }

    public List<BalancePartnerContract> getRawPartnerContracts(long clientId) {
        List<BalancePartnerContract> partnerContracts;
        try {
            partnerContracts = balanceService.getPartnerContracts(clientId);
            clientIdCache.put(clientId, partnerContracts);
        } catch (BalanceException e) {
            partnerContracts = clientIdCache.getIfPresent(clientId);
            if (partnerContracts == null) {
                throw e;
            }
        }

        return partnerContracts;
    }

    private static Map<Integer, CollateralType> createCollateralTypeMap() {
        return Map.ofEntries(
                Map.entry(2000,
                        new CollateralType((contract, collateral) -> contract.setCollateralEndDt(collateral.getDt()))),
                Map.entry(2010,
                        new CollateralType((contract, collateral) -> contract.setCollateralNds(collateral.getNds()))),
                Map.entry(2020, new CollateralType(
                        (contract, collateral) -> {
                            contract.setCollateralPartnerPct(collateral.getPartnerPct());
                            contract.setCollateralAgregatorPct(collateral.getAgregatorPct());
                        })
                ),
                Map.entry(2030,
                        new CollateralType(
                                (contract, collateral) -> contract.setCollateralMkbPrice(collateral.getMkbPrice()))
                ),
                Map.entry(2040, new CollateralType()),
                Map.entry(2050,
                        new CollateralType(
                                (contract, collateral) -> contract.setCollateralEndDt(collateral.getEndDt()))
                ),
                Map.entry(2060, new CollateralType()),
                Map.entry(2070, new CollateralType()),
                Map.entry(2080,
                        new CollateralType(
                                (contract, collateral) ->
                                        contract.setCollateralSearchForms(collateral.getSearchForms()))
                ),
                Map.entry(2090, new CollateralType((contract, collateral) -> {
                    contract.setCollateralEndDt(collateral.getEndDt());
                    contract.setCollateralEndReason(collateral.getEndReason());
                })),
                Map.entry(2100,
                        new CollateralType(
                                (contract, collateral) -> contract.setCollateralNds(collateral.getNds()))
                ),
                Map.entry(2110,
                        new CollateralType(
                                (contract, collateral) -> contract.setCollateralPayTo(collateral.getPayTo()))
                ),
                Map.entry(3010,
                        new CollateralType(
                                (contract, collateral) -> contract.setCollateralNds(collateral.getNds()))
                ),
                Map.entry(3020,
                        new CollateralType(
                                (contract, collateral) -> contract.setCollateralPartnerPct(collateral.getPartnerPct()))
                ),
                Map.entry(3030, new CollateralType((contract, collateral) -> {
                    contract.setCollateralDistributionPlaces(collateral.getDistributionPlaces());
                    contract.setCollateralProductSearch(collateral.getProductSearch());
                    contract.setCollateralProductSearchf(collateral.getProductSearchf());
                    contract.setCollateralProductOptions(collateral.getProductOptions());
                })),
                Map.entry(3040, new CollateralType()),
                Map.entry(3060, new CollateralType((contract, collateral) -> {
                    contract.setCollateralEndDt(collateral.getEndDt());
                    contract.setCollateralTailTime(collateral.getTailTime());
                })),
                Map.entry(3070, new CollateralType((contract, collateral) -> {
                    contract.setCollateralProductsDownload(collateral.getProductsDownload());
                    contract.setCollateralDownloadDomains(collateral.getDownloadDomains());
                })),
                Map.entry(3080, new CollateralType((contract, collateral) -> {
                    contract.setCollateralInstallPrice(collateral.getInstallPrice());
                    contract.setCollateralInstallSoft(collateral.getInstallSoft());
                })),
                Map.entry(3090, new CollateralType((contract, collateral) -> {
                    contract.setCollateralPartnerPct(collateral.getPartnerPct());
                    contract.setCollateralRewardType(collateral.getRewardType());
                }))
        );
    }

    private BalanceContract toBalanceContract(Contract contract) {
        BalanceContract balanceContract = contract.toBalanceContract()
                .withStatus(contract.getContractStatus(messages));

        if (Objects.equals(contract.getContractType(), CoreConstants.BalanceContractType.NEW_OFFER)) {
            balanceContract.withIsOferta1(1);
        }

        if (Objects.equals(contract.getContractType(), CoreConstants.BalanceContractType.OFFER)) {
            if (Objects.equals(CoreConstants.FIRM_ID_YANDEX_EUROPE_AG, contract.getFirm())) {
                balanceContract.withIsOferta1(1);
            } else {
                balanceContract.withIsOferta2(1);
            }
        }

        if (contract.allowsToFillPart2()) {
            balanceContract.withAllowsToFillPart2(true);
        }

        return balanceContract;
    }

    private static class CollateralType {
        private BiConsumer<Contract, Collateral> changeFields;

        CollateralType() {
            this((contract, collateral) -> {
            });
        }

        CollateralType(BiConsumer<Contract, Collateral> changeFields) {
            this.changeFields = changeFields;
        }

        void changeFields(Contract contract, Collateral collateral) {
            changeFields.accept(contract, collateral);
        }
    }
}
