package ru.yandex.partner.core.service.msf;

import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.web.reactive.function.BodyInserters;
import org.springframework.web.reactive.function.client.WebClient;
import org.springframework.web.util.UriComponentsBuilder;

import ru.yandex.partner.core.entity.block.type.commonshowvideoandstrategy.SiteVersionType;
import ru.yandex.partner.core.service.msf.dto.DefaultFormatDto;
import ru.yandex.partner.core.service.msf.dto.FormatGroupDto;
import ru.yandex.partner.core.service.msf.dto.FormatWithSettingsDto;
import ru.yandex.partner.core.service.msf.dto.MsfValidationResultDto;
import ru.yandex.partner.defaultconfiguration.CacheNames;
import ru.yandex.partner.defaultconfiguration.rpc.RpcConfigProps;

import static ru.yandex.partner.libs.common.PartnerRequestService.retryWithFixedDelay;
import static ru.yandex.partner.libs.common.PartnerRequestService.serviceUnavailable;

@ParametersAreNonnullByDefault
@Service
public class FormatSystemServiceImpl implements FormatSystemService {
    private static final Logger logger = LoggerFactory.getLogger(FormatSystemServiceImpl.class);

    private static final String FORMATS_PATH = "/schemas/pcode-7012/formats";
    private static final String VALIDATE_PATH = "/schemas/pcode-7012/validate";
    private static final String P_CODE_SETTINGS_VALIDATE_PATH = "/schemas/pcode-7012/validate-pcode-settings";
    private static final String DEFAULT_FORMAT_PATH = "/default-formats";

    private static final Map<String, String> MSF_SITE_VERSION = Map.of(
            SiteVersionType.TURBO_DESKTOP.getLiteral(), "turbo-desktop",
            SiteVersionType.MOBILE_FULLSCREEN.getLiteral(), "mobile-fullscreen",
            SiteVersionType.MOBILE_REWARDED.getLiteral(), "mobile-rewarded",
            SiteVersionType.MOBILE_FLOORAD.getLiteral(), "mobile-floorad");

    private static final String FORM_PARAM = "form";
    private static final String LANG_PARAM = "lang";
    private static final String ROLE_PARAM = "role";
    private static final String SITE_PARAM = "site";
    private static final String IS_ADFOX = "is_adfox";

    private final ObjectMapper objectMapper;
    private final String msfUrl;
    private final WebClient webClient;
    private final RpcConfigProps rpcConfigProps;

    @Autowired
    public FormatSystemServiceImpl(WebClient webClient,
                                   ObjectMapper objectMapper,
                                   MsfRpcConfig msfRpcConfig) {
        this.webClient = webClient;
        this.objectMapper = objectMapper;
        this.msfUrl = msfRpcConfig.getUrl();
        this.rpcConfigProps = msfRpcConfig;
    }

    @Cacheable(value = {CacheNames.MSF_FORM_CACHE}, unless = "#result == null || #result.isEmpty()")
    @Override
    public List<FormatGroupDto> getFormats(String form, String lang, String userRole, String site) {

        UriComponentsBuilder uriBuilder = UriComponentsBuilder
                .fromUriString(msfUrl)
                .path(FORMATS_PATH)
                .queryParam(FORM_PARAM, form)
                .queryParam(LANG_PARAM, lang)
                .queryParam(ROLE_PARAM, userRole)
                .queryParam(SITE_PARAM, getMsfSiteVersion(site));

        var result = webClient.get()
                .uri(uriBuilder.build().toUri())
                .accept(MediaType.APPLICATION_JSON)
                .exchangeToMono(clientResponse -> clientResponse.toEntityList(FormatGroupDto.class))
                .transform(retryWithFixedDelay(rpcConfigProps))
                .blockOptional()
                .orElse(serviceUnavailable());

        if (result.getStatusCode() != HttpStatus.OK) {
            logger.error("Could not get formats from MSF for params form: {}, role: {}, site: {}, code: {}",
                    form, userRole, site, result.getStatusCodeValue());
            return List.of();
        }

        return result.getBody();
    }

    @Cacheable(value = {CacheNames.MSF_FORMATS_CACHE}, unless = "#result == null")
    @Override
    public FormatWithSettingsDto getFormatSettings(String formatName, String lang, String userRole, String site) {
        UriComponentsBuilder uriBuilder = UriComponentsBuilder
                .fromUriString(msfUrl)
                .path(FORMATS_PATH)
                .pathSegment(formatName)
                .queryParam(LANG_PARAM, lang)
                .queryParam(ROLE_PARAM, userRole)
                .queryParam(SITE_PARAM, getMsfSiteVersion(site));

        var result = webClient.get()
                .uri(uriBuilder.build().toUri())
                .accept(MediaType.APPLICATION_JSON)
                .exchangeToMono(clientResponse -> clientResponse.toEntity(String.class))
                .transform(retryWithFixedDelay(rpcConfigProps))
                .blockOptional()
                .orElse(serviceUnavailable());

        if (result.getStatusCode() != HttpStatus.OK) {
            logger.warn("Could not get format from MSF params format: {}, role: {}, site: {}, code: {}",
                    formatName, userRole, site, result.getStatusCodeValue());
            return null;
        } else {
            try {
                return objectMapper.readValue(result.getBody(), FormatWithSettingsDto.class);
            } catch (JsonProcessingException e) {
                throw new RuntimeException("Cannot parse MSF Response to FormatWithSettingsDto.class. " +
                        "ResponseBody: " + result.getBody(), e);
            }
        }
    }


    @Cacheable(value = {CacheNames.MSF_VALIDATION_CACHE}, unless = "#result == null")
    @Override
    public MsfValidationResultDto validate(String lang, String userRole, String site, Map<String, Object> data) {
        UriComponentsBuilder uriBuilder = UriComponentsBuilder
                .fromUriString(msfUrl)
                .path(VALIDATE_PATH)
                .queryParam(LANG_PARAM, lang)
                .queryParam(ROLE_PARAM, userRole)
                .queryParam(SITE_PARAM, getMsfSiteVersion(site));
        String body = null;
        try {
            body = objectMapper.writeValueAsString(data);
        } catch (JsonProcessingException e) {
            logger.error("Couldn't parse data to json", e);
            return null;
        }
        var result = getValidationResp(uriBuilder, body);

        if (result.getStatusCode() != HttpStatus.OK) {
            logger.error("Could not validate MSF format, response code: {}", result.getStatusCodeValue());
            return null;
        }
        return result.getBody();
    }

    private ResponseEntity<MsfValidationResultDto> getValidationResp(UriComponentsBuilder uriBuilder, String body) {
        return webClient.post()
                .uri(uriBuilder.build().toUri())
                .contentType(MediaType.APPLICATION_JSON)
                .body(BodyInserters.fromValue(body))
                .exchangeToMono(clientResponse -> clientResponse.toEntity(MsfValidationResultDto.class))
                .transform(retryWithFixedDelay(rpcConfigProps))
                .blockOptional()
                .orElse(serviceUnavailable());
    }

    @Cacheable(value = {CacheNames.MSF_P_CODE_SETTINGS_VALIDATION_CACHE}, unless = "#result == null")
    @Override
    public MsfValidationResultDto validatePCodeSettings(String lang, String userRole, String site, Boolean isAdfox,
                                                        Map<String, Object> data) {

        UriComponentsBuilder uriBuilder = UriComponentsBuilder
                .fromUriString(msfUrl)
                .path(P_CODE_SETTINGS_VALIDATE_PATH)
                .queryParam(LANG_PARAM, lang)
                .queryParam(ROLE_PARAM, userRole)
                .queryParam(SITE_PARAM, getMsfSiteVersion(site))
                .queryParam(IS_ADFOX, isAdfox);
        try {
            String body = objectMapper.writeValueAsString(data);
            var result = getValidationResp(uriBuilder, body);

            if (result.getStatusCode() != HttpStatus.OK) {
                logger.error("Could not validate MSF pCodeSettings, response code: {}", result.getStatusCodeValue());
                return null;
            }
            return result.getBody();
        } catch (JsonProcessingException e) {
            logger.error("Couldn't parse data to json", e);
            return null;
        }
    }

    @Cacheable(value = {CacheNames.MSF_DEFAULT_CACHE}, unless = "#result == null")
    @Override
    @Nullable
    public DefaultFormatDto getDefaultFormats(String lang, String userRole, String siteVersion, Boolean isAdfox) {
        UriComponentsBuilder uriBuilder = UriComponentsBuilder
                .fromUriString(msfUrl)
                .path(DEFAULT_FORMAT_PATH)
                .queryParam(ROLE_PARAM, userRole)
                .queryParam(LANG_PARAM, lang)
                .queryParam(SITE_PARAM, getMsfSiteVersion(siteVersion))
                .queryParam(IS_ADFOX, isAdfox);

        var result = webClient.get()
                .uri(uriBuilder.build().toUri())
                .accept(MediaType.APPLICATION_JSON)
                .exchangeToMono(clientResponse -> clientResponse.toEntity(DefaultFormatDto.class))
                .transform(retryWithFixedDelay(rpcConfigProps))
                .blockOptional()
                .orElse(serviceUnavailable());

        if (result.getStatusCode() != HttpStatus.OK || result.getBody() == null) {
            logger.error("Could not get default formats from MSF for site: {}, code: {}",
                    siteVersion, result.getStatusCodeValue());
            return null;
        }

        return result.getBody();
    }

    private String getMsfSiteVersion(String siteVersion) {
        return MSF_SITE_VERSION.getOrDefault(siteVersion, siteVersion);
    }
}
