package ru.yandex.partner.core.service.statevalidation.block;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.Lists;
import net.logstash.logback.marker.LogstashBasicMarker;
import one.util.streamex.StreamEx;
import org.joda.time.Duration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.Marker;
import org.springframework.context.support.DelegatingMessageSource;
import org.springframework.context.support.MessageSourceAccessor;

import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.validation.presentation.DefectPresentationRegistry;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.misc.time.Stopwatch;
import ru.yandex.partner.core.block.BlockUniqueIdConverter;
import ru.yandex.partner.core.entity.QueryOpts;
import ru.yandex.partner.core.entity.block.container.BlockContainer;
import ru.yandex.partner.core.entity.block.container.BlockContainerImpl;
import ru.yandex.partner.core.entity.block.filter.BlockFilters;
import ru.yandex.partner.core.entity.block.model.BaseBlock;
import ru.yandex.partner.core.entity.block.service.BlockService;
import ru.yandex.partner.core.entity.block.service.BlockValidationService;
import ru.yandex.partner.core.entity.block.service.OperationMode;
import ru.yandex.partner.core.entity.page.filter.PageFilters;
import ru.yandex.partner.core.entity.page.model.BasePage;
import ru.yandex.partner.core.entity.page.model.ContextPage;
import ru.yandex.partner.core.entity.page.model.PageWithCommonFields;
import ru.yandex.partner.core.entity.page.service.PageService;
import ru.yandex.partner.core.filter.CoreFilterNode;
import ru.yandex.partner.core.filter.meta.MetaFilter;
import ru.yandex.partner.core.filter.operator.BinaryOperator;
import ru.yandex.partner.core.filter.operator.FilterOperator;
import ru.yandex.partner.core.service.statevalidation.StateValidationService;
import ru.yandex.partner.core.utils.OrderBy;
import ru.yandex.partner.dbschema.partner.enums.ContextOnSiteRtbModel;
import ru.yandex.partner.libs.i18n.TranslatableError;

public abstract class BlockStateValidationService<B extends BaseBlock, P extends BasePage>
        implements StateValidationService {

    private static final Logger logger = LoggerFactory.getLogger(BlockStateValidationService.class);
    private static final Duration ALLOWED_ITERATION_DURATION = Duration.standardMinutes(1);

    private final BlockValidationService blockValidationService;
    private final BlockService blockService;
    private final PageService pageService;
    private final int threadsCount;
    private final int step;
    private final MetaFilter<BaseBlock, Long> blockPageIdMetaFilter;
    private final MetaFilter<BasePage, Long> pageIdMetaFilter;
    private final DefectPresentationRegistry<TranslatableError> defectRegistry;
    private final MessageSourceAccessor messageSourceAccessor;
    private final ObjectMapper objectMapper;

    public BlockStateValidationService(BlockValidationService blockValidationService,
                                       BlockService blockService,
                                       PageService pageService,
                                       int threadsCount,
                                       int step,
                                       DefectPresentationRegistry<TranslatableError> defectRegistry,
                                       ObjectMapper objectMapper) {
        this.blockValidationService = blockValidationService;
        this.blockService = blockService;
        this.pageService = pageService;
        this.threadsCount = threadsCount;
        this.step = step;
        this.defectRegistry = defectRegistry;
        this.objectMapper = objectMapper;
        this.messageSourceAccessor = new MessageSourceAccessor(new DelegatingMessageSource(), Locale.ENGLISH);
        this.blockPageIdMetaFilter = BlockFilters.PAGE_ID;
        this.pageIdMetaFilter = PageFilters.PAGE_ID;
    }

    @Override
    public CronValidationResultDto validateObjects(int remainder) {


        return validateObjects(remainder, pageService.count(QueryOpts.forClass(getPageClass())
                .withFilter(CoreFilterNode.neutral())));
    }

    @Override
    public CronValidationResultDto validateObjects(int remainder, Supplier<Boolean> shouldContinue) {
        return validateObjects(remainder, pageService.count(
                QueryOpts.forClass(getPageClass())
                        .withFilter(CoreFilterNode.neutral())), shouldContinue);
    }

    public CronValidationResultDto validateObjects(int remainder, long pageCount, Supplier<Boolean> shouldContinue) {
        return validate(remainder, pageCount, shouldContinue);
    }

    public CronValidationResultDto validateObjects(int remainder, long pageCount) {
        return validate(remainder, pageCount, () -> true);
    }

    /**
     * for tests and oneshot
     *
     * @param pageIds
     */
    public void validateByPageIds(List<Long> pageIds) {
        var container = BlockContainerImpl.create(OperationMode.CRON);
        blockValidationService.forceFillContainerByValidationSupports(container);
        Lists.partition(pageIds, step).forEach(part -> {

            CoreFilterNode<BaseBlock> coreFilter = prepareBlockFilter(part);

            var blocks = blockService.findAll(QueryOpts.forClass(getBlockClass())
                    .withFilter(coreFilter)
                    .withProps(getModelProperties())
            );
            validateBlocks(blocks, container);
        });
    }

    private CronValidationResultDto validate(int remainder, long pageCount, Supplier<Boolean> shouldContinue) {
        long pageValidationCount = 0L;
        long lastId = 0;

        OrderBy order = new OrderBy(PageWithCommonFields.PAGE_ID);
        LimitOffset limit = new LimitOffset(step, 0);

        BlockContainer containerWithAllDict = BlockContainerImpl.create(OperationMode.CRON);
        blockValidationService.forceFillContainerByValidationSupports(containerWithAllDict);

        var blocksCount = 0;
        Map<Long, ValidationResult<?, Defect>> defects = new HashMap<>();

        var interrupted = false;

        while (pageValidationCount <= pageCount) {
            if (!shouldContinue.get()) {
                logger.warn("validation has been interrupted, completed partially {}/{}",
                        pageValidationCount, pageCount);
                interrupted = true;
                break;
            }
            var stopwatch = Stopwatch.createAndStart();

            var pages = getPagesNextPart(lastId, limit, order);
            if (pages.isEmpty()) {
                break;
            }
            logger.debug("page validation count {}/{}", pageValidationCount, pageCount);
            var size = pages.size();
            pageValidationCount += size;
            lastId = pages.get(size - 1).getId();
            var pageIds = pages.stream().map(BasePage::getId).collect(Collectors.toList());
            if (threadsCount > 1) {
                pageIds = pageIds.stream().filter(it -> it % threadsCount == remainder)
                        .collect(Collectors.toList());
            }

            var coreFilter = prepareBlockFilter(pageIds);

            var blocks = blockService.findAll(QueryOpts.forClass(getBlockClass())
                    .withFilter(coreFilter)
                    .withProps(getModelProperties())
            );
            blocksCount += blocks.size();
            var curDefects = proceedValidation(blocks, containerWithAllDict);

            defects.putAll(curDefects);

            stopwatch.stop();
            var iterationDuration = stopwatch.duration();

            if (iterationDuration.isLongerThan(ALLOWED_ITERATION_DURATION)) {
                logger.warn("validation iteration for {} pages and {} blocks completed in {} seconds",
                        pages.size(),
                        blocks.size(),
                        iterationDuration.getStandardSeconds());
            } else {
                logger.trace("validation iteration for {} pages and {} blocks completed in {} seconds",
                        pages.size(),
                        blocks.size(),
                        iterationDuration.getStandardSeconds());
            }
        }
        if (!defects.isEmpty()) {
            List<String> mappedDef = mapDefects(defects);
            logDefects(mappedDef, blocksCount);
        }
        logger.info("Thread #{} validation finished {}", remainder, interrupted ? "partially" : "fully");
        return new CronValidationResultDto(defects.size(), blocksCount, getValidatedModelName(), interrupted);

    }

    private List<P> getPagesNextPart(Long lastId, LimitOffset limit, OrderBy order) {
        return pageService.findAll(QueryOpts.forClass(getPageClass())
                .withFilter(CoreFilterNode.create(pageIdMetaFilter, FilterOperator.GREATER, lastId))
                .withProps(Set.of(ContextPage.ID, PageWithCommonFields.PAGE_ID))
                .withLimitOffset(limit)
                .withOrder(order)
        );
    }

    private Map<Long, ValidationResult<?, Defect>> proceedValidation(List<B> blocks,
                                                                     BlockContainer containerWithAllDict) {
        int size = blocks.size();
        Map<Long, ValidationResult<?, Defect>> defects = new HashMap<>();

        for (int i = 0; i < size; i += this.step) {
            int to = Math.min(i + step - 1, size);
            logger.debug("Block Part validation size: {}", to - i);
            var result = blockValidationService.validateAsCron(blocks.subList(i, to),
                    containerWithAllDict);
            logger.debug("Part finished");
            if (result.hasAnyErrors()) {
                var blockIds = result.getValue().stream().map(BaseBlock::getId).collect(Collectors.toList());
                defects.putAll(result.getSubResults().entrySet()
                        .stream().filter(it -> it.getValue().hasAnyErrors())
                        .collect(Collectors.toMap(it -> blockIds.get(Integer.parseInt(it.getKey().toString())),
                                Map.Entry::getValue)));
            }
        }
        return defects;
    }

    private void logDefects(List<String> defects, int blocksCount) {
        Marker marker = new LogstashBasicMarker("Java validation cron");
        logger.error(marker, "Java - {} Total problem elements {}/{}", getValidatedModelName(),
                defects.size(), blocksCount);

        for (var error : defects) {
            logger.error(marker, "Validation failed for block: {}", error);
        }
    }

    private ArrayList<String> mapDefects(Map<Long, ValidationResult<?, Defect>> defects) {
        var data = StreamEx.of(defects.entrySet()).map(entry ->
                new ValidationDefectDto(entry.getKey(),
                        BlockUniqueIdConverter.convertToPublicId(entry.getKey()),
                        entry.getValue().flattenErrors().stream().map(it ->
                                new ValidationDefectDto.ValidationErrorDto(
                                        it.getPath().getNodes().stream()
                                                .map(Object::toString).collect(Collectors.joining("/")),
                                        messageSourceAccessor.getMessage(defectRegistry
                                                .getPresentation(it).getMessage()),
                                        it.getValue())
                        ).collect(Collectors.toList()))).toList();
        var result = new ArrayList<String>();
        for (var elt : data) {
            try {
                result.add(objectMapper.writeValueAsString(elt));
            } catch (JsonProcessingException e) {
                e.printStackTrace();
            }
        }
        return result;
    }

    private CoreFilterNode<BaseBlock> prepareBlockFilter(List<Long> pageIds) {
        var filter = CoreFilterNode.in(blockPageIdMetaFilter, pageIds);
        var addFilter = prepareAdditionalBlockFilters();
        return filter.addFilterNode(BinaryOperator.AND, addFilter);
    }

    protected abstract ContextOnSiteRtbModel getModel();

    protected abstract Class<B> getBlockClass();

    protected abstract Class<P> getPageClass();

    protected abstract CoreFilterNode<BaseBlock> prepareAdditionalBlockFilters();

    protected abstract Set<ModelProperty<? extends Model, ?>> getModelProperties();

    public abstract String getValidatedModelName();

    private void validateBlocks(List<B> blocks, BlockContainerImpl container) {
        var defects = proceedValidation(blocks, container);
        for (var error : mapDefects(defects)) {
            logger.error(error);
        }
    }
}
