package ru.yandex.partner.core.service.yt;

import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.databind.JsonNode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.web.util.UriComponentsBuilder;

import ru.yandex.inside.yt.kosher.impl.YtConfiguration;
import ru.yandex.inside.yt.kosher.impl.YtUtils;
import ru.yandex.inside.yt.kosher.impl.common.YtServiceUnavailableException;
import ru.yandex.inside.yt.kosher.tables.types.JacksonTableEntryType;
import ru.yandex.partner.core.configuration.YTProperties;

@ParametersAreNonnullByDefault
@Service
public class YTServiceImpl implements YTService {
    private static final Logger logger = LoggerFactory.getLogger(YTServiceImpl.class);

    private final YTProperties ytProperties;
    private final ConcurrentHashMap<String, YtConfiguration> ytSelectConfigurationMap;

    @Autowired
    public YTServiceImpl(YTProperties ytProperties) {
        this.ytProperties = ytProperties;
        // нет метода Collectors.toConcurrentMap(Map.Entry::getKey, Map.Entry::getValue, ConcurrentHashMap::new)
        // есть только с mergeFunction, чтоб не зависеть от внутренней реализации Collectors.toConcurrentMap сделал так
        this.ytSelectConfigurationMap = new ConcurrentHashMap<>(
                ytProperties.getHosts()
                        .stream()
                        .map(host -> Map.entry(host, makeConfiguration(host)))
                        .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue)));
    }

    @Override
    public <T> List<T> selectFromAnyReplicas(String yql, Function<Iterator<JsonNode>, List<T>> callback) {
        for (var host : ytProperties.getHosts()) {
            try {
                var result = selectRows(host, yql, callback);
                if (Objects.nonNull(result) && !result.isEmpty()) {
                    return result;
                }
            } catch (YtServiceUnavailableException e) {
                logger.warn("YT service unavailable on host {}", host);
            }

        }
        logger.error("Yt service returned empty result");
        return List.of();
    }

    private <T> List<T> selectRows(String host, String yql, Function<Iterator<JsonNode>, List<T>> callback) {
        var ytConfig = ytSelectConfigurationMap.computeIfAbsent(host, this::makeConfiguration);

        return YtUtils.http(ytConfig).tables().selectRows(yql, new JacksonTableEntryType(), callback);
    }

    private YtConfiguration makeConfiguration(String host) {
        UriComponentsBuilder uriBuilder = UriComponentsBuilder
                .fromUriString("http://" + host + "/api/v3/select_rows");

        return YtConfiguration.builder().
                withToken(ytProperties.getToken()).
                withApiUrl(uriBuilder.toUriString()).
                withSimpleCommandsRetries(3).
                build();
    }
}
