package ru.yandex.partner.core.utils;

import java.io.IOException;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.JsonNodeFactory;

@SuppressWarnings("HideUtilityClassConstructor")
public class CommonConverters {
    public static final Long TRUE = 1L;
    public static final Long FALSE = 0L;

    public static Long booleanToLong(Boolean bool) {
        return bool != null ? (bool ? TRUE : FALSE) : null;
    }

    public static Boolean booleanFromLong(Long l) {
        return l != null ? (l == 1L) : null;
    }

    public static <T> List<T> jsonNodeToList(ObjectMapper objectMapper, JsonNode value, Class<T> entryType) {
        if (value == null || value.isNull() || value.isMissingNode()) {
            return null;
        }
        try {
            List<T> result = new ArrayList<>(value.size());
            for (JsonNode node : value) {
                result.add(objectMapper.treeToValue(node, entryType));
            }
            return result;
        } catch (JsonProcessingException | ClassCastException e) {
            throw new IllegalStateException("Invalid value in JsonNode", e);
        }
    }

    public static Map<String, Object> jsonNodeToMap(ObjectMapper objectMapper, JsonNode value) {
        if (value == null || value.isNull() || value.isMissingNode()) {
            return null;
        }
        try {
            return objectMapper.readValue(value.toString(), new TypeReference<>() {
            });
        } catch (JsonProcessingException | ClassCastException e) {
            throw new IllegalStateException("Invalid value in JsonNode", e);
        }
    }

    public static <T> JsonNode listToJsonNode(ObjectMapper objectMapper, List<T> value) {
        return objectMapper.valueToTree(value);
    }

    public static <T> JsonNode mapToJsonNode(ObjectMapper objectMapper, Map<String, T> value) {
        return objectMapper.valueToTree(value);
    }

    public static <T> List<T> jsonStringToListThrowable(ObjectMapper objectMapper, String value, Class<T> entryType)
            throws IOException {
        if (value == null) {
            return null;
        }

        if (value.length() == 0) {
            return Collections.emptyList();
        }
        return objectMapper.readValue(value,
                objectMapper.getTypeFactory().constructCollectionType(List.class, entryType));
    }

    public static <T> List<T> jsonStringToList(ObjectMapper objectMapper, String value, Class<T> entryType) {
        if (value == null) {
            return null;
        }

        if (value.length() == 0) {
            return Collections.emptyList();
        }

        try {
            return jsonStringToListThrowable(objectMapper, value, entryType);
        } catch (IOException e) {
            throw new IllegalStateException(String.format("Invalid value in json: %s", value), e);
        }
    }

    public static <T> String listToJsonString(ObjectMapper objectMapper, List<T> value) {
        if (value == null) {
            return null;
        }

        try {
            return objectMapper.writeValueAsString(value);
        } catch (IOException e) {
            throw new IllegalStateException("Can not serialize to json", e);
        }

    }

    public static <T> String listToPrettyJsonString(ObjectMapper objectMapper, List<T> value) {
        // для целей совместного тестирования перл/джава а то флапают тесты
        if (value == null) {
            return null;
        } else if (value.isEmpty()) {
            return "[]";
        }

        try {
            return objectMapper.writerWithDefaultPrettyPrinter().writeValueAsString(value);
        } catch (IOException e) {
            throw new IllegalStateException("Can not serialize to json", e);
        }

    }

    public static BigDecimal jsonNodeToBigDecimal(JsonNode value) {
        if (value == null || value.isNull() || value.isMissingNode()) {
            return null;
        }
        try {
            return new BigDecimal(value.asText());
        } catch (NumberFormatException e) {
            throw new IllegalStateException("Invalid value in JsonNode", e);
        }
    }

    public static JsonNode bigDecimalToJsonNode(BigDecimal v) {
        return v == null ? null : JsonNodeFactory.instance.numberNode(v);
    }

}
