package ru.yandex.partner.core.utils;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.function.Supplier;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableList;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;

/**
 * Выполняет дедубликацию переданного списка, возвращая в результате, список уникальных элементов исходного списка
 * и информацию о позициях дубликатов в исходном списке, позволяющую восстановить расположение всех элементов.
 * <p>
 * Например, для исходного списка [7, 8, 8, 7] вернётся результат с уникальными элементами
 * [7, 8] и информацией о позициях [[0, 3], [1, 2]], в которых они встречаются в исходном списке.
 */
@ParametersAreNonnullByDefault
public class Deduplicator {
    private Deduplicator() {
    }

    /**
     * Выполнить дедубликацию на списке, элементы которого можно сравнить с помощью заданного {@code comparator}.
     * Если на элементах списка опеределена хэш функция, то следует использовать метод {@link #deduplicate(List)}
     *
     * @param input      исходный список, подлежащий дедубликации
     * @param comparator сравниватель элементов исходного списка
     * @param <T>        тип элементов исходного списка
     * @return результат дедубликации {@link ru.yandex.partner.core.utils.Deduplicator.Result}
     */
    @Nonnull
    public static <T> Result<T> deduplicate(List<T> input, Comparator<T> comparator) {
        return deduplicate(input, () -> new TreeMap<>(comparator));
    }

    /**
     * Выполнить дедубликацию на списке, на элементах которого задана хэш-функция
     * Используется {@link HashMap} для определения дубликатов.
     *
     * @param input исходный список, подлежащий дедубликации
     * @param <T>   тип элементов исходного списка
     * @return результат дедубликации {@link ru.yandex.partner.core.utils.Deduplicator.Result}
     */
    @SuppressWarnings({"Convert2MethodRef", "WeakerAccess"})
    @Nonnull
    public static <T> Result<T> deduplicate(List<T> input) {
        return deduplicate(input, () -> new HashMap<>());
    }

    @Nonnull
    private static <T> Result<T> deduplicate(List<T> input, Supplier<Map<T, List<Integer>>> mapSupplier) {
        // уникальное значение -> список позиций
        Map<T, List<Integer>> valueToPositions = mapSupplier.get();
        EntryStream.of(input).forKeyValue(
                (index, element) -> valueToPositions.computeIfAbsent(element, k -> new ArrayList<>()).add(index));

        ImmutableList<List<Integer>> deduplicationInfo = StreamEx.of(valueToPositions.values())
                // сортируем по первому вхождению для того, чтобы значения
                // встречались в том же порядке, что и в исходном списке
                .sortedBy(positions -> positions.get(0))
                .map(ImmutableList::copyOf)
                .collect(ImmutableList.toImmutableList());

        ImmutableList<T> deduplicated = deduplicationInfo.stream()
                .map(positions -> positions.get(0))
                .map(input::get)
                .collect(ImmutableList.toImmutableList());

        return new Result<>(deduplicated, deduplicationInfo);
    }

    public static class Result<T> {
        private final List<T> deduplicated;
        private final List<List<Integer>> deduplicationInfo;

        private Result(ImmutableList<T> deduplicated, ImmutableList<List<Integer>> deduplicationInfo) {
            this.deduplicated = deduplicated;
            this.deduplicationInfo = deduplicationInfo;
        }

        /**
         * @return уникальные элементы исходного списка, в порядке их исходной встречаемости
         */
        @Nonnull
        public List<T> getDeduplicated() {
            return deduplicated;
        }

        /**
         * @return информация о позициях уникальных элементов в исходном массиве (список параллельный c deduplicated)
         */
        @Nonnull
        public List<List<Integer>> getDeduplicationInfo() {
            return deduplicationInfo;
        }
    }
}
