package ru.yandex.partner.core.utils.converter;

import java.util.List;
import java.util.function.Function;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@SuppressWarnings("unused")
@FunctionalInterface
public interface Converter<T1, T2> {
    /**
     * Создать конвертер из функции
     *
     * @param function функция конвертации
     * @param <T1>     тип исходного значения
     * @param <T2>     тип значения результата конвертации
     * @return конвертер
     */
    static <T1, T2> Converter<T1, T2> of(Function<T1, T2> function) {
        return function::apply;
    }

    T2 convert(T1 source);

    /**
     * Сконвертировать в список
     */
    default List<T2> convertList(Iterable<T1> sourceIterable) {
        return mapList(sourceIterable, this::convert);
    }

    /**
     * Создать новый конвертер, который применяет nextConverter после текущего
     */
    default <T3> Converter<T1, T3> andThen(Converter<? super T2, T3> nextConverter) {
        return source -> nextConverter.convert(this.convert(source));
    }

    /**
     * Сделать новый конвертер с помощью wrapper, который может использовать текущий
     *
     * @param wrapper функция, принимающия текущий конвертер и делающая с его использованием новый
     */
    default <W1, W2> Converter<W1, W2> wrapTo(Function<Converter<T1, T2>, Converter<W1, W2>> wrapper) {
        return wrapper.apply(this);
    }
}
