package ru.yandex.partner.core.validation;

import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;

import com.google.common.collect.Sets;
import org.jetbrains.annotations.Nullable;

import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;
import ru.yandex.partner.core.validation.defects.PartnerCollectionDefects;

/**
 * Абстрактный валидатор для поля модели, который нужно проверить по справочнику
 * на дублирующиеся или несуществующие id
 *
 * @param <M> - класс модели
 * @param <E> - класс элемента данных, который нужно проверить по справочнику
 */
public abstract class ModelWithDictEntriesListValidator<M extends Model, E> implements Validator<List<M>, Defect> {
    protected final Set<Long> existingIds;

    public ModelWithDictEntriesListValidator(Set<Long> existingIds) {
        this.existingIds = existingIds != null ? existingIds : Set.of();
    }

    @Override
    public ValidationResult<List<M>, Defect> apply(List<M> models) {

        return ListValidationBuilder.<M, Defect>of(models)
                .checkEachBy(this::checkModel)
                .getResult();
    }

    /**
     * Провалидировать модель (помимо проверки по спарвочникам)
     *
     * @param vb
     */
    protected abstract void validateModel(ModelItemValidationBuilder<M> vb);

    /**
     * ModelProperty, в котором лежит список сущностей, которые надо проверить по справочнику
     *
     * @return
     */
    protected abstract ModelProperty<? super M, List<E>> getModelPropertyToCheck();

    /**
     * Функция для получения Id сущности, которая проверяется по справочнику
     *
     * @return
     */
    @Nonnull
    protected abstract Function<E, Long> idGetter();

    protected ValidationResult<M, Defect> checkModel(M model) {
        ModelItemValidationBuilder<M> vb = ModelItemValidationBuilder.of(model);
        if (getModelPropertyToCheck().get(model) != null) {
            validateModel(vb);
            vb.item(getModelPropertyToCheck())
                    .checkByFunction(entries -> checkIds(entries), When.isValid());
        }
        return vb.getResult();
    }

    @Nullable
    protected Defect<Set<Long>> checkIds(List<E> entries) {

        List<Long> ids = entries.stream()
                .map(idGetter()).collect(Collectors.toList());

        LinkedHashSet<Long> uniques = new LinkedHashSet<>();
        LinkedHashSet<Long> duplicateIds =
                ids.stream().filter(e -> !uniques.add(e)).collect(Collectors.toCollection(LinkedHashSet::new));

        if (!duplicateIds.isEmpty()) {
            return PartnerCollectionDefects.duplicateEntries(duplicateIds);
        }

        LinkedHashSet<Long> notExistingIds = new LinkedHashSet<>(Sets.difference(uniques, existingIds));

        if (!notExistingIds.isEmpty()) {
            return PartnerCollectionDefects.entriesNotFound(notExistingIds);
        }

        return null;
    }
}
