package ru.yandex.partner.core.validation.defects.presentation;

import java.util.stream.Collectors;

import ru.yandex.direct.validation.defect.ids.StringDefectIds;
import ru.yandex.direct.validation.presentation.DefaultDefectPresentationRegistry;
import ru.yandex.direct.validation.result.DefectIds;
import ru.yandex.partner.core.action.exception.ActionDefectIds;
import ru.yandex.partner.core.action.exception.DefectInfoWithMsgId;
import ru.yandex.partner.core.action.exception.presentation.ActionDefectMsg;
import ru.yandex.partner.core.validation.defects.ids.ModelPropertyDefectIds;
import ru.yandex.partner.core.validation.defects.ids.PartnerCollectionDefectIds;
import ru.yandex.partner.core.validation.defects.ids.TypeDefectIds;
import ru.yandex.partner.libs.i18n.MsgWithArgs;
import ru.yandex.partner.libs.i18n.TranslatableError;

import static ru.yandex.partner.core.validation.defects.presentation.DefectPresentationProviders.defaultDefectProvider;
import static ru.yandex.partner.core.validation.defects.presentation.DefectPresentationProviders.getProvider;


public class CommonDefectPresentations {
    private CommonDefectPresentations() {
    }

    public static DefaultDefectPresentationRegistry<TranslatableError> commonDefectPresentationRegistry(
            ModelPropertyPresentationNameService modelPropertyPresentationNameService) {

        return DefaultDefectPresentationRegistry.builder()
                .register(
                        TypeDefectIds.INVALID_TYPE,
                        getProvider(
                                defectInfo -> defectInfo.getDefect().params().getGettextMsg()
                        )
                )
                .register(DefectIds.CANNOT_BE_NULL, getProvider(
                        defectInfo ->
                                MsgWithArgs.of(
                                        CommonValidationMsg.DATA_CANNOT_BE_NULL,
                                        defectInfo.getPath().getFieldName()
                                )
                ))
                .register(DefectIds.MUST_BE_NULL, defaultDefectProvider(CommonValidationMsg.DATA_MUST_BE_NULL))
                .register(TypeDefectIds.DATA_MUST_BE_INTEGER_NUMBER,
                        defaultDefectProvider(CommonValidationMsg.DATA_MUST_BE_INTEGER_NUMBER))
                .register(TypeDefectIds.DATA_MUST_BE_BOOLEAN,
                        defaultDefectProvider(CommonValidationMsg.DATA_MUST_BE_BOOLEAN))
                .register(TypeDefectIds.DATA_MUST_BE_DEFINED,
                        defaultDefectProvider(CommonValidationMsg.DATA_MUST_BE_DEFINED))
                .register(TypeDefectIds.DATA_MUST_BE_HEXADECIMAL,
                        defaultDefectProvider(CommonValidationMsg.INCORRECT_JSON))
                .register(TypeDefectIds.DATA_MUST_BE_NUMERIC,
                        defaultDefectProvider(CommonValidationMsg.DATA_MUST_BE_NUMBER))
                .register(StringDefectIds.LENGTH_CANNOT_BE_LESS_THAN_MIN,
                        getProvider(
                                defectInfo ->
                                        MsgWithArgs.of(
                                                CommonValidationMsg.LENGTH_CANNOT_BE_LESS_THAN_MIN,
                                                defectInfo.getValue(),
                                                defectInfo.getDefect().params().getMinLength()
                                        )
                        ))
                .register(StringDefectIds.LENGTH_CANNOT_BE_MORE_THAN_MAX,
                        getProvider(
                                defectInfo ->
                                        MsgWithArgs.of(
                                                CommonValidationMsg.LENGTH_MORE_THAN,
                                                defectInfo.getValue(),
                                                defectInfo.getDefect().params().getMaxLength()
                                        )
                        )
                )
                .register(PartnerCollectionDefectIds.Size.MUST_BE_IN_COLLECTION_WITH_PRESENTATION,
                        getProvider(
                                defectInfo ->
                                        MsgWithArgs.of(
                                                CommonValidationMsg.DATA_MUST_BE_IN_COLLECTION,
                                                defectInfo.getValue(),
                                                defectInfo.getDefect().params().getStringPresentation()
                                        )
                        ))
                .register(PartnerCollectionDefectIds.Entries.DUPLICATE_ENTRIES,
                        getProvider(
                                defectInfo ->
                                        MsgWithArgs.of(
                                                CommonValidationMsg.DUPLICATE_ENTRIES,
                                                defectInfo.getDefect().params().stream()
                                                        .map(Object::toString)
                                                        .collect(Collectors.joining(", "))
                                        )
                        ))
                .register(PartnerCollectionDefectIds.Entries.ENTRIES_NOT_FOUND,
                        getProvider(
                                defectInfo ->
                                        MsgWithArgs.of(
                                                CommonValidationMsg.ENTRIES_NOT_FOUND,
                                                defectInfo.getDefect().params().stream()
                                                        .map(Object::toString)
                                                        .collect(Collectors.joining(", "))
                                        )
                        ))
                .register(ActionDefectIds.ActionDefect.CAN_NOT_DO_ACTION,
                        getProvider(
                                defectInfo -> MsgWithArgs.of(
                                        ActionDefectMsg.CAN_NOT_DO_ACTION,
                                        defectInfo.getDefect().params().getActionName()))
                )
                .register(ActionDefectIds.ActionDefect.INCORRECT_REQUEST_BODY,
                        getProvider(defectInfo -> MsgWithArgs.of(ActionDefectMsg.INCORRECT_REQUEST_BODY)))
                .register(ModelPropertyDefectIds.FIELD_SHOULD_BE_POSITIVE_NUMBER,
                        getProvider(
                                defectInfo ->
                                        MsgWithArgs.of(
                                                CommonValidationMsg.FIELD_SHOULD_BE_POSITIVE_NUMBER,
                                                modelPropertyPresentationNameService.getPresentationName(
                                                        defectInfo.getDefect().params().getModelClass(),
                                                        defectInfo.getDefect().params().getModelProperty()
                                                )
                                        )
                        ))
                .register(DefectInfoWithMsgId.ID,
                        getProvider(defectInfo -> defectInfo.getDefect().params().getGettextMsg()))
                .register(DefectIds.INVALID_VALUE, defaultDefectProvider(CommonValidationMsg.INVALID_VALUE))
                .build();
    }
}
