package ru.yandex.partner.dbschemagen;

import java.nio.file.Path;
import java.sql.Connection;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;

import org.jooq.meta.jaxb.ForcedType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.dbschemagen.DbSchemaGen;
import ru.yandex.direct.dbschemagen.JooqDbInfo;
import ru.yandex.direct.dbschemagen.Util;
import ru.yandex.direct.dbschemagen.VcsWorker;
import ru.yandex.direct.dbschemagen.VcsWorkerFactory;
import ru.yandex.direct.jcommander.ParserWithHelp;
import ru.yandex.direct.mysql.MySQLDockerContainer;
import ru.yandex.direct.process.Docker;
import ru.yandex.direct.process.Processes;
import ru.yandex.direct.test.mysql.DirectMysqlDb;
import ru.yandex.direct.test.mysql.TestMysqlConfig;
import ru.yandex.direct.utils.GracefulShutdownHook;
import ru.yandex.direct.utils.io.TempDirectory;

public class DbSchemaUpdatePartner {
    private static final Logger logger = LoggerFactory.getLogger(DbSchemaUpdatePartner.class);

    private static final String[] DB_NAMES = {
            "partner"
    };

    private static final String DEFAULT_IMAGE = "registry.yandex.net/partners/yandex-partners-percona:5.7.34-2";

    private static final TestMysqlConfig MYSQL_CONFIG = new TestMysqlConfig(
            "partner/java/libs/test-db", "mysql-server", "mysql-test-data",
            "/ru/yandex/partner/test/db/dbschema_docker_image.txt",
            "partner/java/libs/test-db/src/main/resources/ru/yandex/partner/test/db/dbschema_docker_image.txt"
    );

    /**
     * Набор столбцов, генерируемых как {@link org.jooq.impl.SQLDataType#BIGINT}/{@link Long}
     */
    private static final String[] ARTIFICIAL_BIGINT_COLUMNS = {
    };

    /**
     * Набор столбцов, генерируемых как {@link org.jooq.impl.SQLDataType#BIGINTUNSIGNED}/{@link org.jooq.types.ULong}.
     */
    private static final String[] ARTIFICIAL_BIGINTUNSIGNED_COLUMNS = {
    };

    /**
     * Набор столбцов, генерируемых как {@link org.jooq.impl.SQLDataType#SMALLINTUNSIGNED}/{@link org.jooq.types.UShort}
     */
    private static final String[] ARTIFICIAL_SMALLINTUNSIGNED_COLUMNS = {
    };

    private DbSchemaUpdatePartner() {
    }

    public static void main(String[] args) throws Exception {
        VcsWorker vcsWorker = VcsWorkerFactory.getWorker();

        DbSchemaUpdateParams params = new DbSchemaUpdateParams();
        ParserWithHelp.parse(DbSchemaUpdatePartner.class.getCanonicalName(), args, params);

        Path wcRoot = vcsWorker.getWorkingCopyRoot();
        if (!wcRoot.resolve(".arcadia.root").toFile().exists()) {
            Processes.exitWithFailure(
                    "Something is wrong with your arcadia checkout. Could not find file `.arcadia.root`."
            );
        }

        Path dbSchemaLibPath = wcRoot.resolve("partner/java/libs/dbschema");
        Path dbSchemaDir = wcRoot.resolve("partner/java/libs/dbschema/src/main/resources");

        DirectMysqlDb testDb = new DirectMysqlDb(MYSQL_CONFIG);

        try (
                GracefulShutdownHook ignored = new GracefulShutdownHook(Duration.ofMinutes(1));
                MySQLDockerContainer mysql = new MySQLDockerContainer(testDb.docker().createRunner(new Docker(),
                        DEFAULT_IMAGE));
                TempDirectory tmpDir = new TempDirectory("DbSchemaUpdateSvn")
        ) {
            /* ГЕНЕРАЦИЯ JOOQ-КЛАССОВ */

            List<ForcedType> forcedTypes = getForcedTypes();

            DbSchemaGen generator = new DbSchemaGen("ru.yandex.partner.dbschema", new JooqDbInfo(mysql))
                    .withForcedTypes(forcedTypes);

            try {
                for (Path schemaSqlDir : Arrays.stream(DB_NAMES).map(dbSchemaDir::resolve)
                        .collect(Collectors.toList())) {
                    String schemaName = schemaSqlDir.getFileName().toString();
                    try (Connection conn = mysql.connect(Duration.ofSeconds(20), Util::setMultiQueries)) {
                        DbSchemaGen.createSchema(conn, schemaName);
                        conn.setCatalog(schemaName);
                        DbSchemaGen.populate(schemaSqlDir, conn, false);
                        generator.generate(schemaName, dbSchemaLibPath.resolve("src/main/java"));
                    }
                }
            } catch (Exception exc) {
                logger.error("Generation failed, keeping SQL schema dir for further investigation: " + dbSchemaDir);
                tmpDir.keep();
                throw exc;
            }

            // Регистрируем изменения в vcs
            vcsWorker.registerChanges(dbSchemaLibPath);

            mysql.stop();

            if (params.createImage || params.forceUpload) {

                String imageTag = "registry.yandex.net/partners/dbschema" + ":" + generateImageTag(mysql);
                logger.info("Snapshotting docker image " + imageTag + "...");
                testDb.docker().createImage(wcRoot, mysql, imageTag);

                if (params.forceUpload) {
                    logger.info("Uploading docker image " + imageTag + "...");
                    mysql.getDocker().push(imageTag);

                    if (params.forceMysqlServerUpload) {
                        logger.info("Uploading mysql server to sandbox...");
                        testDb.sandbox().uploadMysqlServer(wcRoot, mysql);
                    }

                    logger.info("Uploading mysql data to sandbox...");
                    testDb.sandbox().uploadMysqlData(wcRoot, mysql);
                }

            }

        }
    }

    @Nonnull
    private static String generateImageTag(MySQLDockerContainer mysql) throws InterruptedException {
        return mysql.getImage()
                .replace("registry.yandex.net/partners/", "")
                .replace(':', '-')
                + "--" + Instant.now().getEpochSecond();
    }

    private static List<ForcedType> getForcedTypes() {
        Collection<ForcedType> forcedTypesLong =
                createForcedTypes(ARTIFICIAL_BIGINT_COLUMNS, "BIGINT");

        Collection<ForcedType> forcedTypesULong =
                createForcedTypes(ARTIFICIAL_BIGINTUNSIGNED_COLUMNS, "BIGINTUNSIGNED");

        Collection<ForcedType> forcedTypesUShort =
                createForcedTypes(ARTIFICIAL_SMALLINTUNSIGNED_COLUMNS, "SMALLINTUNSIGNED");

        List<ForcedType> forcedTypes = new ArrayList<>(forcedTypesLong);
        forcedTypes.addAll(forcedTypesULong);
        forcedTypes.addAll(forcedTypesUShort);
        Set<String> uniqExpressions =
                forcedTypes.stream().map(ForcedType::getExpression).collect(Collectors.toSet());
        if (uniqExpressions.size() != forcedTypesLong.size() + forcedTypesULong.size() + forcedTypesUShort.size()) {
            throw new IllegalStateException("Looks like there is overlapping in forced types. " +
                    "Please review your changes");
        }
        forcedTypes.add(new ForcedType()
                .withName("BIGINT")
                .withIncludeTypes("BIGINT(\\(\\d+\\))? UNSIGNED")
        );
        return forcedTypes;
    }

    /**
     * Генерирует набор {@link ForcedType}, переопределяющих sql-тип.
     *
     * @param columns           набор переопределяемых столбцов
     * @param forcedSqlTypeName необходимый sql-тип. Тип следует брать из {@link org.jooq.DataType}-констант
     *                          класса {@link org.jooq.impl.SQLDataType}
     * @return коллекцию настроенных ForcedTypes
     */
    private static Collection<ForcedType> createForcedTypes(String[] columns, String forcedSqlTypeName) {

        return Arrays.stream(columns)
                .map(column -> new ForcedType()
                        .withName(forcedSqlTypeName)
                        .withExpression(Pattern.quote(column))
                ).collect(Collectors.toList());
    }
}
