package ru.yandex.partner.defaultconfiguration;

import java.sql.SQLException;

import javax.servlet.http.HttpServletRequest;
import javax.sql.DataSource;

import org.jooq.DSLContext;
import org.jooq.SQLDialect;
import org.jooq.VisitListenerProvider;
import org.jooq.conf.Settings;
import org.jooq.impl.DefaultVisitListenerProvider;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Primary;
import org.springframework.transaction.annotation.EnableTransactionManagement;
import org.springframework.web.context.annotation.RequestScope;

import ru.yandex.partner.defaultconfiguration.healthchecks.DBCheck;
import ru.yandex.partner.defaultconfiguration.jooq.QueryListener;

@Configuration
@EnableTransactionManagement
public class MysqlConfiguration {
    public static final String MYSQL_DATASOURCE_BEAN_NAME = "mysqlDataSource";

    @Bean
    public MysqlProps mysqlProps() {
        return new MysqlProps();
    }

    @Bean
    @ConditionalOnProperty(name = "mysql.scope", havingValue = "singleton", matchIfMissing = true)
    public DBCheck dbCheck(DSLContext dslContext) {
        return new DBCheck(dslContext);
    }

    @Primary
    @Bean(name = MYSQL_DATASOURCE_BEAN_NAME)
    @ConditionalOnProperty(name = "mysql.scope", havingValue = "singleton", matchIfMissing = true)
    DataSource mysqlDataSourceSingleton(
            MysqlProps mysqlProps
    ) throws SQLException {
        return MysqlUtils.createDataSource(
                mysqlProps.getHost(),
                mysqlProps.getPort(),
                mysqlProps.getSchema(),
                mysqlProps.getParams(),
                mysqlProps.getUsername(),
                mysqlProps.getPassword(),
                mysqlProps.getConnectionTimeout(),
                mysqlProps.getLockWaitTimeout(),
                mysqlProps.getMaxPoolSize()
        );
    }


    @Bean
    Settings settings(ObjectProvider<JooqSettingsConfigurer> configurers) {
        var settings = new Settings()
                .withRenderSchema(false)
                .withFetchWarnings(false)
                .withExecuteLogging(true)
                .withQueryTimeout(5 * 60) // 5 min
                .withRenderGroupConcatMaxLenSessionVariable(false)
                .withInterpreterDialect(SQLDialect.MYSQL);
        configurers.iterator().forEachRemaining(cfgr -> cfgr.configure(settings));
        return settings;
    }

    @Bean
    VisitListenerProvider visitListenerProvider(
            @Value("${application.name}") String applicationName) {
        return new DefaultVisitListenerProvider(new QueryListener(applicationName));
    }

    @Primary
    @Bean(name = MYSQL_DATASOURCE_BEAN_NAME, destroyMethod = "fakeClose")
    @RequestScope
    @ConditionalOnProperty(name = "mysql.scope", havingValue = "request")
    DataSource mysqlDataSourceRequestScope(
            HttpServletRequest request,
            MysqlConfigureService mysqlConfigureService
    ) {
        if (mysqlConfigureService.hasConfigurerData(request)) {
            return mysqlConfigureService.createDataSource(request);
        } else {
            return mysqlConfigureService.createDataSource();
        }
    }

    @Bean
    @ConditionalOnProperty(name = "mysql.scope", havingValue = "request")
    public MysqlConfigureService mysqlConfigurerService(
            MysqlProps mysqlProps,
            @Value("${mysql.cache.size:10}") long size
    ) {
        return new MysqlConfigureService(mysqlProps, size);
    }
}
