package ru.yandex.partner.defaultconfiguration;

import java.io.IOException;
import java.sql.SQLException;
import java.util.Objects;

import javax.servlet.http.HttpServletRequest;
import javax.sql.DataSource;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.RemovalListener;
import com.zaxxer.hikari.HikariConfig;
import com.zaxxer.hikari.HikariDataSource;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpStatus;
import org.springframework.web.server.ResponseStatusException;


public class MysqlConfigureService {
    public static final Logger LOGGER = LoggerFactory.getLogger(MysqlConfigureService.class);
    private static final String HEADER_NAME = "Mysql-Configuration";
    private final Cache<ConnectionProperties, HikariDataSource> cache;
    private final ObjectMapper objectMapper;
    private final MysqlProps mysqlProps;

    public MysqlConfigureService(MysqlProps mysqlProps, long cacheSize) {
        this.mysqlProps = mysqlProps;
        this.objectMapper = new ObjectMapper();

        cache = CacheBuilder.newBuilder()
                .maximumSize(cacheSize)
                .removalListener((RemovalListener<ConnectionProperties, HikariDataSource>) notification -> {
                    LOGGER.warn("DataSource close notification: {}", notification);
                    notification.getValue().close();
                })
                .build();
    }

    public boolean hasConfigurerData(HttpServletRequest request) {
        String dbConfig = request.getHeader(HEADER_NAME);
        return dbConfig != null && !dbConfig.isBlank();
    }

    public DataSource createDataSource(HttpServletRequest request) {
        String dbConfig = request.getHeader(HEADER_NAME);
        try {
            var connectionProperties = objectMapper.readValue(dbConfig, ConnectionProperties.class);
            return createDataSource(connectionProperties);
        } catch (IOException e) {
            throw new ResponseStatusException(HttpStatus.BAD_REQUEST, "Cannot parse properties from " +
                    HEADER_NAME + " header. Current value = " + dbConfig, e);
        }
    }

    public DataSource createDataSource() {
        var connectionProperties = new ConnectionProperties(
                mysqlProps.getHost(),
                mysqlProps.getPort(),
                mysqlProps.getSchema(),
                mysqlProps.getUsername(),
                mysqlProps.getPassword()
        );
        return createDataSource(connectionProperties);
    }

    private DataSource createDataSource(ConnectionProperties connectionProperties) {
        try {
            return cache.get(connectionProperties,
                    () -> new FakeClosableHikariDataSource(makeConfig(connectionProperties)));
        } catch (Exception e) {
            throw new ResponseStatusException(HttpStatus.INTERNAL_SERVER_ERROR, e.getMessage(), e);
        }
    }

    private HikariConfig makeConfig(ConnectionProperties connectionProperties) throws SQLException {
        return MysqlUtils.createHikariConfig(connectionProperties.getMysqlHost(),
                connectionProperties.getMysqlPort(),
                connectionProperties.getSchema(),
                mysqlProps.getParams(),
                connectionProperties.getUsername(),
                connectionProperties.getPassword(),
                mysqlProps.getConnectionTimeout(),
                mysqlProps.getLockWaitTimeout(),
                mysqlProps.getMaxPoolSize()
        );
    }

    @SuppressWarnings("unused")
    private static class ConnectionProperties {
        @JsonProperty("mysql_host")
        private String mysqlHost;
        @JsonProperty("mysql_port")
        private String mysqlPort;
        private String schema;
        private String username;
        private String password;

        public ConnectionProperties() {
            // for ObjectMapper
        }

        public ConnectionProperties(String mysqlHost, String mysqlPort, String schema, String username,
                                    String password) {
            this.mysqlHost = mysqlHost;
            this.mysqlPort = mysqlPort;
            this.schema = schema;
            this.username = username;
            this.password = password;
        }

        public String getMysqlHost() {
            return mysqlHost;
        }

        public void setMysqlHost(String mysqlHost) {
            this.mysqlHost = mysqlHost;
        }

        public String getMysqlPort() {
            return mysqlPort;
        }

        public void setMysqlPort(String mysqlPort) {
            this.mysqlPort = mysqlPort;
        }

        public String getSchema() {
            return schema;
        }

        public void setSchema(String schema) {
            this.schema = schema;
        }

        public String getUsername() {
            return username;
        }

        public void setUsername(String username) {
            this.username = username;
        }

        public String getPassword() {
            return password;
        }

        public void setPassword(String password) {
            this.password = password;
        }

        @Override
        public String toString() {
            return "ConnectionProperties{" +
                    "mysqlHost='" + mysqlHost + '\'' +
                    ", mysqlPort='" + mysqlPort + '\'' +
                    ", schema='" + schema + '\'' +
                    ", username='" + username + '\'' +
                    ", password='" +
                    StringUtils.repeat('*', Objects.requireNonNullElse(password, "").length()) + '\'' +
                    '}';
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (!(o instanceof ConnectionProperties)) {
                return false;
            }
            ConnectionProperties that = (ConnectionProperties) o;
            return Objects.equals(mysqlHost, that.mysqlHost)
                    && Objects.equals(mysqlPort, that.mysqlPort)
                    && Objects.equals(schema, that.schema)
                    && Objects.equals(username, that.username)
                    && Objects.equals(password, that.password);
        }

        @Override
        public int hashCode() {
            return Objects.hash(mysqlHost, mysqlPort, schema, username, password);
        }
    }
}
