package ru.yandex.partner.defaultconfiguration;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.time.Duration;

import javax.sql.DataSource;

import com.google.common.base.Strings;
import com.mysql.cj.jdbc.MysqlDataSource;
import com.mysql.cj.jdbc.exceptions.CommunicationsException;
import com.zaxxer.hikari.HikariConfig;
import com.zaxxer.hikari.HikariDataSource;

public class MysqlUtils {

    private MysqlUtils() {
        // utils
    }

    public static DataSource createDataSource(
            String host, String port, String schema, String params,
            String username,
            String password,
            Duration connectionTimeout,
            Duration lockWaitTimeout,
            int maxPoolSize
    ) throws SQLException {
        return new HikariDataSource(createHikariConfig(host, port, schema, params, username, password,
                connectionTimeout, lockWaitTimeout, maxPoolSize));
    }

    public static HikariConfig createHikariConfig(
            String host, String port, String schema,
            String params,
            String username, String password,
            Duration connectionTimeout,
            Duration lockWaitTimeout,
            int maxPoolSize
    ) throws SQLException {
        MysqlDataSource mysqlDataSource = new MysqlDataSource();
        mysqlDataSource.setUrl(String.format("jdbc:mysql://%s:%s/%s?%s", host, port, schema, params));
        mysqlDataSource.setUser(username);
        mysqlDataSource.setPassword(password);

        try (Connection connection = mysqlDataSource.getConnection();
             PreparedStatement preparedStatement = connection.prepareStatement("SET sql_mode = ''")) {
            preparedStatement.execute();
        } catch (CommunicationsException e) {
            throw new RuntimeException(
                    "Can't connect to DB: host=%s, port=%s, schema=%s, username=%s, password=%s".formatted(
                            host, port, schema, username, Strings.repeat("*", password.length())
                    ),
                    e
            );
        }

        return setupCommonHikariConfig(mysqlDataSource, connectionTimeout, lockWaitTimeout, maxPoolSize);
    }

    public static HikariConfig setupCommonHikariConfig(
            MysqlDataSource mysqlDataSource,
            Duration connectionTimeout,
            Duration lockWaitTimeout,
            int maxPoolSize
    ) {
        HikariConfig hikariConfig = new HikariConfig();
        hikariConfig.setConnectionTimeout(connectionTimeout.toMillis());
        hikariConfig.setConnectionInitSql(
                "SET @@LOCAL.innodb_lock_wait_timeout = " + lockWaitTimeout.getSeconds() + ";"
        );
        hikariConfig.setDataSource(mysqlDataSource);
        hikariConfig.setMaximumPoolSize(maxPoolSize);
        return hikariConfig;
    }
}
