package ru.yandex.partner.defaultconfiguration;

import java.util.function.Consumer;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.databind.ObjectMapper;
import io.netty.channel.ChannelOption;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.http.MediaType;
import org.springframework.http.client.reactive.ReactorClientHttpConnector;
import org.springframework.http.client.reactive.ReactorResourceFactory;
import org.springframework.http.codec.ClientCodecConfigurer;
import org.springframework.http.codec.json.Jackson2JsonDecoder;
import org.springframework.web.reactive.function.client.ExchangeStrategies;
import org.springframework.web.reactive.function.client.WebClient;

@ParametersAreNonnullByDefault
@Configuration
public class WebClientConfiguration {

    @Bean
    public WebClient webClient(WebClient.Builder builder, ObjectMapper objectMapper) {
        // билдер уже сконфигурирован бутом в части метрик и кодеков
        var textPlainDecoder = new Jackson2JsonDecoder(objectMapper, MediaType.TEXT_PLAIN);
        textPlainDecoder.setMaxInMemorySize(20 * 1024 * 1024);
        Consumer<ClientCodecConfigurer> c = configurer ->
                configurer.customCodecs()
                        .register(textPlainDecoder);
        builder
                .exchangeStrategies(ExchangeStrategies.builder().codecs(c).build());

        return builder.build();
    }

    @Configuration
    @ConditionalOnClass(reactor.netty.http.client.HttpClient.class)
    public static class ReactorNetty {

        @Bean
        @ConditionalOnMissingBean
        public ReactorResourceFactory reactorClientResourceFactory() {
            return new ReactorResourceFactory();
        }

        @Bean
        public ReactorClientHttpConnector reactorClientHttpConnector(
                ReactorResourceFactory reactorResourceFactory) {
            return new ReactorClientHttpConnector(
                    reactorResourceFactory,
                    client -> client.option(ChannelOption.CONNECT_TIMEOUT_MILLIS, 60 * 1000)
            );
        }
    }

}
