package ru.yandex.partner.defaultconfiguration.jooq;

import com.google.gson.Gson;
import org.jooq.Clause;
import org.jooq.VisitContext;
import org.jooq.impl.DefaultVisitListener;
import org.slf4j.MDC;

import ru.yandex.partner.defaultconfiguration.PartnerLocalDateTime;
import ru.yandex.partner.defaultconfiguration.logging.ErrorBoosterMessageConverter;

import java.time.format.DateTimeFormatter;
import java.util.EnumSet;

public class QueryListener extends DefaultVisitListener {
    private final DateTimeFormatter formatter;
    private final Gson serializer;
    private final String applicationName;
    private final EnumSet<Clause> clausesSet;

    public QueryListener(String applicationName) {
        this.applicationName = applicationName;
        this.serializer = new Gson();
        this.formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");
        this.clausesSet = getClausesSet();
    }

    @Override
    public void clauseStart(VisitContext ctx) {
        if (clausesSet.contains(ctx.clause()) && !ctx.context().subquery()) {
            String comment = getJsonComment(getQueryCommentObject());
            ctx.context().sql(comment);
        }
    }

    private EnumSet<Clause> getClausesSet() {
        return EnumSet.of(
                Clause.SELECT,
                Clause.INSERT,
                Clause.DELETE,
                Clause.UPDATE
        );
    }

    private QueryComment getQueryCommentObject() {
        return  new QueryComment(
                MDC.get(ErrorBoosterMessageConverter.MDCKeys.YANDEX_LOGIN),
                applicationName,
                applicationName.equals("java_hourglass")
                        ? MDC.get(ErrorBoosterMessageConverter.MDCKeys.CRON_METHOD)
                        : MDC.get(ErrorBoosterMessageConverter.MDCKeys.REQUEST_METHOD) + MDC.get(ErrorBoosterMessageConverter.MDCKeys.REQUEST_PATH),
                PartnerLocalDateTime.now().format(formatter)
        );
    }

    private String getJsonComment(QueryComment comment) {
        return "/* " + serializer.toJson(comment) + " */";
    }

    private static class QueryComment {
        public String login;
        public String system;
        public String method;
        public String time;

        public QueryComment(String login, String system, String method, String time) {
            this.login = login;
            this.system = system;
            this.method = method;
            this.time = time;
        }
    }
}
