package ru.yandex.partner.defaultconfiguration.logging;

import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

import ch.qos.logback.classic.spi.ILoggingEvent;
import ch.qos.logback.classic.spi.IThrowableProxy;
import ch.qos.logback.classic.spi.ThrowableProxy;
import ch.qos.logback.classic.spi.ThrowableProxyUtil;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.protobuf.ByteString;
import org.slf4j.MDC;

import ru.yandex.logbroker.agent.logback.appender.MessageConverter;
import ru.yandex.partner.libs.exceptions.FingerPrintable;

public class ErrorBoosterMessageConverter implements MessageConverter {
    // Не очень красиво, но пока не нашел другого способа прокинуть сюда это значение
    private static String env = "unknown";
    private static String source = "java_unknown";
    private final ObjectMapper objectMapper;

    @SuppressWarnings("unused")
    public ErrorBoosterMessageConverter() {
        this(new ObjectMapper());
        // need for logback
    }

    ErrorBoosterMessageConverter(ObjectMapper objectMapper) {
        this.objectMapper = objectMapper;
    }

    @Override
    public ByteString convert(ILoggingEvent event) {
        return ByteString.copyFromUtf8(convert0(event));
    }

    private String convert0(ILoggingEvent event) {
        JsonModel jsonModel = new JsonModel()
                .withEnv(env)
                .withSource(source)
                .withDc(getDc())
                .withHost(getHost())
                .withFingerprint(getFingerPrint(event))
                .withMessage(event.getFormattedMessage())
                .withTimestamp(event.getTimeStamp())
                .withStack(stackTraceToString(event))
                .withLevel(event.getLevel().toString().toLowerCase());

        String yandexUidStr = MDC.get(MDCKeys.YANDEX_UID);
        if (yandexUidStr != null) {
            long yandexUid;
            try {
                yandexUid = Long.parseLong(yandexUidStr);
            } catch (NumberFormatException e) {
                yandexUid = -1L;
            }
            jsonModel.withYandexuid(yandexUid);
        }

        String requestPath = MDC.get(MDCKeys.REQUEST_PATH);
        if (requestPath != null) {
            jsonModel.withAdditional("requestpath", requestPath);
        }

        String requestMethod = MDC.get(MDCKeys.REQUEST_METHOD);
        if (requestMethod != null) {
            jsonModel.withAdditional("requestmethod", requestMethod);

        }

        try {
            return objectMapper.writeValueAsString(jsonModel);
        } catch (JsonProcessingException e) {
            throw new ErrorBoosterException("Cannot convert to json", e);
        }
    }

    private String getDc() {
        // Пока только для DEPLOY
        return Optional.ofNullable(System.getenv("DEPLOY_NODE_DC")).orElse("");
    }

    private String getHost() {
        // Пока только для DEPLOY
        return Optional.ofNullable(System.getenv("DEPLOY_POD_PERSISTENT_FQDN")).orElse("");
    }

    private String getFingerPrint(ILoggingEvent event) {
        String fingerPrint = tryFingerPrintable(event);
        if (fingerPrint != null) {
            return fingerPrint;
        }

        StackTraceElement[] stackTraceElements = event.getCallerData();

        if (stackTraceElements != null && stackTraceElements.length > 0) {
            return stackTraceElements[0].toString();
        } else {
            return null;
        }
    }

    private String tryFingerPrintable(ILoggingEvent event) {
        if (event.getThrowableProxy() instanceof ThrowableProxy) {
            Throwable throwable = ((ThrowableProxy) event.getThrowableProxy()).getThrowable();
            if (throwable instanceof FingerPrintable) {
                return ((FingerPrintable) throwable).getFingerPrint();
            }
        }

        return null;
    }

    private String stackTraceToString(ILoggingEvent event) {
        IThrowableProxy throwableProxy = event.getThrowableProxy();
        if (throwableProxy != null) {
            return ThrowableProxyUtil.asString(throwableProxy);
        }
        return "";
    }

    public static void setErrorBoosterEnvBySpringProfile(String springProfile) {
        switch (springProfile) {
            case "prod":
                env = "production";
                break;
            case "test":
            case "autotest":
                env = "testing";
                break;
            default:
                env = "development";
                break;
        }
    }

    public static void setErrorApplicationSource(String source) {
        ErrorBoosterMessageConverter.source = source;
    }

    public static class MDCKeys {
        public static final String YANDEX_UID = "yandex_uid";
        public static final String REQUEST_PATH = "request_path";
        public static final String REQUEST_METHOD = "request_method";
        public static final String YANDEX_LOGIN = "yandex_login";
        public static final String CRON_METHOD = "cron_method";
        public static final String CRON_PATH = "cron_path";

        private MDCKeys() {
            // constants only
        }
    }

    @SuppressWarnings({"UnusedReturnValue", "unused"})
    private static class JsonModel {
        private String env;
        private String source;
        private String dc;
        private String host;
        @JsonInclude(JsonInclude.Include.NON_NULL)
        private String fingerprint;
        private String message;
        private long timestamp;
        private String stack;
        private String level;
        @JsonInclude(JsonInclude.Include.NON_NULL)
        private Long yandexuid;

        @JsonInclude(JsonInclude.Include.NON_EMPTY)
        private Map<String, String> additional = new HashMap<>();

        public JsonModel withEnv(String env) {
            this.env = env;
            return this;
        }

        public JsonModel withSource(String source) {
            this.source = source;
            return this;
        }

        public JsonModel withDc(String dc) {
            this.dc = dc;
            return this;
        }

        public JsonModel withHost(String host) {
            this.host = host;
            return this;
        }

        public JsonModel withFingerprint(String fingerprint) {
            this.fingerprint = fingerprint;
            return this;
        }

        public JsonModel withMessage(String message) {
            this.message = message;
            return this;
        }

        public JsonModel withTimestamp(long timestamp) {
            this.timestamp = timestamp;
            return this;
        }

        public JsonModel withStack(String stack) {
            this.stack = stack;
            return this;
        }

        public JsonModel withLevel(String level) {
            this.level = level;
            return this;
        }

        public JsonModel withYandexuid(Long yandexuid) {
            this.yandexuid = yandexuid;
            return this;
        }

        public JsonModel withAdditional(String paramName, String paramValue) {
            additional.put(paramName, paramValue);
            return this;
        }

        public String getLanguage() {
            return "java";
        }

        public String getProject() {
            return "partner";
        }

        public String getService() {
            return "java_backend";
        }

        public String getEnv() {
            return env;
        }

        public String getSource() {
            return source;
        }

        public String getDc() {
            return dc;
        }

        public String getHost() {
            return host;
        }

        public String getFingerprint() {
            return fingerprint;
        }

        public String getMessage() {
            return message;
        }

        public long getTimestamp() {
            return timestamp;
        }

        public String getStack() {
            return stack;
        }

        public String getLevel() {
            return level;
        }

        public Long getYandexuid() {
            return yandexuid;
        }

        public Map<String, String> getAdditional() {
            return additional;
        }
    }
}
