package ru.yandex.partner.defaultconfiguration.logging;

import java.io.IOException;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;

import javax.annotation.ParametersAreNonnullByDefault;

import ch.qos.logback.classic.pattern.ThrowableProxyConverter;
import ch.qos.logback.classic.spi.ILoggingEvent;
import com.fasterxml.jackson.core.JsonGenerator;
import com.google.common.collect.Maps;
import net.logstash.logback.composite.ContextJsonProvider;
import net.logstash.logback.composite.JsonWritingUtils;
import org.slf4j.MDC;

@ParametersAreNonnullByDefault
public class PartnersJsonProvider<Event extends ILoggingEvent> extends ContextJsonProvider<Event> {

    private final DateTimeFormatter logDateTimeFormatter;
    private final ThrowableProxyConverter throwableProxyConverter;
    private final int paramsSize;

    public PartnersJsonProvider() {
        String hostname;
        try {
            hostname = InetAddress.getLocalHost().getHostName();
        } catch (UnknownHostException e) {
            hostname = "UNKNOWN";
        }
        MDC.put(PID_NAME, String.valueOf(ProcessHandle.current().pid()));
        MDC.put(HOST, hostname);
        MDC.put(STAGE_NAME, System.getenv("DEPLOY_STAGE_ID"));
        MDC.put(UNIT_NAME, System.getenv("DEPLOY_UNIT_ID"));
        MDC.put(APP_NAME, System.getenv("SYSTEM"));

        logDateTimeFormatter = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");
        throwableProxyConverter = new ThrowableProxyConverter();
        paramsSize = LogParams.values().length;
    }

    @Override
    public void start() {
        throwableProxyConverter.start();
        super.start();
    }

    @Override
    public void writeTo(JsonGenerator generator, Event event) throws IOException {
        var mdcPropertyMap = event.getMDCPropertyMap();
        var log = Maps.newHashMapWithExpectedSize(mdcPropertyMap.size() + paramsSize);
        var dateTime = LocalDateTime.ofInstant(Instant.ofEpochMilli(event.getTimeStamp()), ZoneId.systemDefault());
        String className = null;
        String methodName = null;
        String errorData = null;
        String stackTrace = null;

        var callerData = event.getCallerData();
        if (event.hasCallerData() && callerData.length > 0) {
            var firstCaller = callerData[callerData.length - 1];
            className = firstCaller.getClassName();
            methodName = firstCaller.getMethodName();
        }

        if (event.getThrowableProxy() != null) {
            errorData = event.getThrowableProxy().getMessage();
            stackTrace = throwableProxyConverter.convert(event);
        }

        log.put(LogParams.DATE_TIME.getParamName(), dateTime.format(this.logDateTimeFormatter));
        log.put(LogParams.LEVEL.getParamName(), event.getLevel().levelStr);
        log.put(LogParams.THREAD.getParamName(), event.getThreadName());
        log.put(LogParams.CLASS.getParamName(), className);
        log.put(LogParams.METHOD.getParamName(), methodName);
        log.put(LogParams.MESSAGE.getParamName(), event.getFormattedMessage());
        log.put(LogParams.ERROR.getParamName(), errorData);
        log.put(LogParams.STACK_TRACE.getParamName(), stackTrace);
        log.putAll(event.getMDCPropertyMap());

        JsonWritingUtils.writeMapEntries(generator, log);
    }

    private static final String PID_NAME = "pid";
    private static final String STAGE_NAME = "stage_name";
    private static final String UNIT_NAME = "unit_name";
    private static final String APP_NAME = "app_name";
    private static final String HOST = "hostname";

    private enum LogParams {
        DATE_TIME("datetime"),
        LEVEL("level"),
        THREAD("thread"),
        CLASS("class"),
        METHOD("method"),
        MESSAGE("message"),
        ERROR("error"),
        STACK_TRACE("stackTrace");

        private final String paramName;

        LogParams(String paramName) {
            this.paramName = paramName;
        }

        public String getParamName() {
            return paramName;
        }
    }
}
