package ru.yandex.partner.libs.auth.configuration;

import java.util.List;

import com.fasterxml.jackson.databind.ObjectMapper;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.security.authentication.AuthenticationManager;
import org.springframework.security.web.context.NullSecurityContextRepository;
import org.springframework.security.web.context.SecurityContextPersistenceFilter;
import org.springframework.web.reactive.function.client.WebClient;
import org.springframework.web.servlet.mvc.method.annotation.RequestMappingHandlerMapping;

import ru.yandex.partner.core.configuration.CoreConfiguration;
import ru.yandex.partner.libs.auth.manager.DefaultPartnerAuthenticationManager;
import ru.yandex.partner.libs.auth.provider.apikey.ApiKeyAuthenticationProvider;
import ru.yandex.partner.libs.auth.provider.apikey.ApiKeyRpcConfig;
import ru.yandex.partner.libs.auth.provider.cookie.CookieAuthenticationProvider;
import ru.yandex.partner.libs.auth.provider.tvm.TvmAuthenticationProvider;
import ru.yandex.partner.libs.auth.service.LoginUrlService;
import ru.yandex.partner.libs.auth.service.UserDetailsService;
import ru.yandex.partner.libs.extservice.blackbox.BlackboxService;
import ru.yandex.passport.tvmauth.CheckedServiceTicket;
import ru.yandex.passport.tvmauth.TvmClient;

@Configuration
@Import({CoreConfiguration.class})
@ComponentScan({"ru.yandex.partner.libs.auth", "ru.yandex.partner.libs.extservice"})
public class BasicSecurityConfiguration {

    @Bean
    public ApiKeyAuthenticationProvider apiKeyAuthenticationProvider(
            WebClient webClient,
            ObjectMapper objectMapper,
            @Value("${auth.apikey.token:}") String serviceToken,
            ApiKeyRpcConfig apiKeyRpcConfig
    ) {
        return new ApiKeyAuthenticationProvider(serviceToken, webClient, objectMapper, apiKeyRpcConfig);
    }

    @Bean
    public CookieAuthenticationProvider cookieAuthenticationProvider(BlackboxService blackboxService,
                                                                     LoginUrlService loginUrlService) {
        return new CookieAuthenticationProvider(blackboxService, loginUrlService);
    }

    @Bean
    public TvmAuthenticationProvider tvmAuthenticationProvider(@Value("${tvm.selfDstAlias}") String selfDstAlias,
                                                               TvmClient tvmClient) {
        // способ узнать свой id
        String ticket = tvmClient.getServiceTicketFor(selfDstAlias);
        CheckedServiceTicket checkedServiceTicket = tvmClient.checkServiceTicket(ticket);

        if (!checkedServiceTicket.booleanValue()) {
            throw new RuntimeException("You have not self access by TVM. auth.tvm.selfDstAlias: " + selfDstAlias);
        }

        int selfServiceId = checkedServiceTicket.getSrc();
        return new TvmAuthenticationProvider(tvmClient, selfServiceId);
    }

    /**
     * Менеджер аутентификации. Перебирает провайдеров, которые могут аутентифицировать пользователя
     * Используется в AuthenticationFilter
     *
     * @param apiKeyAuthenticationProvider провайдер "кабинет разработчика"
     * @param cookieAuthenticationProvider провайдер "куки/блэкбокс"
     * @param tvmAuthenticationProvider    провайдер "TVM"
     * @param userDetailsService           сервис, который умеет получать данные пользователя из базы
     * @param requestMappingHandlerMapping
     * @return менеджер
     */
    @Bean
    public AuthenticationManager authenticationManager(
            ApiKeyAuthenticationProvider apiKeyAuthenticationProvider,
            CookieAuthenticationProvider cookieAuthenticationProvider,
            TvmAuthenticationProvider tvmAuthenticationProvider,
            UserDetailsService userDetailsService,
            LoginUrlService loginUrlService,
            RequestMappingHandlerMapping requestMappingHandlerMapping) {
        return new DefaultPartnerAuthenticationManager(
                List.of(apiKeyAuthenticationProvider, cookieAuthenticationProvider, tvmAuthenticationProvider),
                userDetailsService, loginUrlService, requestMappingHandlerMapping);
    }

    /**
     * Фильтр, отвечающий за хранение контекстов авторизации между запросами.
     * Так как нам между запросами их хранить не надо, создаём такой, который ничего не хранит.
     * (если не определить этот фильтр, spring security создаст сам и будет хранить контексты)
     * https://clck.ru/MmC96
     *
     * @return фильтр
     */
    @Bean
    public SecurityContextPersistenceFilter securityContextPersistenceFilter() {
        return new SecurityContextPersistenceFilter(new NullSecurityContextRepository());
    }

}
