package ru.yandex.partner.libs.auth.manager

import org.springframework.security.authentication.AuthenticationProvider
import org.springframework.security.core.Authentication
import org.springframework.security.core.AuthenticationException
import org.springframework.web.method.HandlerMethod
import org.springframework.web.servlet.mvc.method.annotation.RequestMappingHandlerMapping
import ru.yandex.partner.libs.auth.annotation.Auth
import ru.yandex.partner.libs.auth.exception.ForceFailureAuthenticationException
import ru.yandex.partner.libs.auth.exception.authentication.NotRegisteredAuthenticationException
import ru.yandex.partner.libs.auth.model.UserAuthenticationHolder
import ru.yandex.partner.libs.auth.provider.ForceFailureAuthenticationProviderAdapter
import ru.yandex.partner.libs.auth.provider.HandlerSupportedAuthTypeProvider
import ru.yandex.partner.libs.auth.service.LoginUrlService
import ru.yandex.partner.libs.auth.service.UserDetailsService
import ru.yandex.partner.libs.exceptions.HttpErrorStatusEnum
import ru.yandex.partner.libs.exceptions.I18nResponseStatusException

/**
 * Версия [PartnerAuthenticationManager], которая бросает 401 если хотя бы один провайдер
 * пробовал аутентифицировать запрос, но не смог (выбросил [AuthenticationException])
 *
 * Здесь также фильтруются способы аутентификации, когда метод контроллера размечен [Auth]
 */
class DefaultPartnerAuthenticationManager(
    providers: List<AuthenticationProvider>,
    userDetailsService: UserDetailsService?,
    loginUrlService: LoginUrlService?,
    private val requestMappingHandlerMapping: RequestMappingHandlerMapping
) : PartnerAuthenticationManager(
    adaptAuthenticationProviders(providers),
    userDetailsService,
    loginUrlService
) {
    /**
     * also see [.internalAuthenticate]
     */
    @Throws(AuthenticationException::class)
    override fun authenticate(authentication: Authentication): Authentication {
        return try {
            internalAuthenticate(authentication)
        } catch (e: NotRegisteredAuthenticationException) {
            // Если есть в пасспорте, но нет в partner2 то вернем 404
            throw I18nResponseStatusException(HttpErrorStatusEnum.ERROR__NOT_FOUND)
        }
    }

    /**
     * Вызывайте этот метод, если нужно вызвать аутентификацию ручками
     * Этот метод не проебразует в HTTP exception-ы
     */
    @Throws(AuthenticationException::class)
    fun internalAuthenticate(authentication: Authentication): Authentication {
        return try {
            resolveSupportedAuthTypes(authentication)
            super.authenticate(authentication)
        } catch (e: ForceFailureAuthenticationException) {
            throw e.authenticationException
        }
    }

    private fun resolveSupportedAuthTypes(authentication: Authentication) {
        if (authentication !is UserAuthenticationHolder) return

        val request = authentication.httpServletRequest
        val handler = requestMappingHandlerMapping.getHandler(request)?.handler

        if (handler !is HandlerMethod) {
            return
        }

        request.setAttribute(
            HandlerSupportedAuthTypeProvider.SUPPORTED_AUTH_TYPES,
            handler.getMethodAnnotation(Auth::class.java)?.value?.toSet()
        )
    }

    companion object {
        private fun adaptAuthenticationProviders(
            providers: List<AuthenticationProvider>,
        ): List<AuthenticationProvider> {
            return providers
                .map { ForceFailureAuthenticationProviderAdapter(it) }
                .map { HandlerSupportedAuthTypeProvider(it) }
        }
    }
}
