package ru.yandex.partner.libs.auth.manager;

import java.util.List;

import javax.servlet.http.HttpServletRequest;

import org.springframework.security.authentication.AuthenticationManager;
import org.springframework.security.authentication.AuthenticationProvider;
import org.springframework.security.authentication.ProviderManager;
import org.springframework.security.authentication.ProviderNotFoundException;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.AuthenticationException;

import ru.yandex.partner.libs.auth.exception.authentication.NoAuthenticationDataException;
import ru.yandex.partner.libs.auth.message.AuthErrorMsg;
import ru.yandex.partner.libs.auth.model.UserAuthentication;
import ru.yandex.partner.libs.auth.model.UserAuthenticationHolder;
import ru.yandex.partner.libs.auth.service.LoginUrlService;
import ru.yandex.partner.libs.auth.service.UserDetailsService;

/**
 * Менеджер аутентификации. Помимо стандартного поведения {@link ProviderManager}
 * 1) Насыщает объект аутентификации кастомными данными через {@link UserDetailsService}
 * 2) Переопределяет текст ошибки {@link ProviderNotFoundException}
 * которая возникает если ни один из провайдеров не стал обрабатывать запрос
 * <p>
 * Эта версия провайдера, как и {@link ProviderManager}, продолжает перебирать провайдеров,
 * если один из них выбросил {@link AuthenticationException}
 */
public class PartnerAuthenticationManager extends ProviderManager {

    private final UserDetailsService userDetailsService;
    private final LoginUrlService loginUrlService;

    public PartnerAuthenticationManager(List<AuthenticationProvider> providers,
                                        UserDetailsService userDetailsService, LoginUrlService loginUrlService) {
        super(providers);
        this.userDetailsService = userDetailsService;
        this.loginUrlService = loginUrlService;
    }

    public PartnerAuthenticationManager(List<AuthenticationProvider> providers, AuthenticationManager parent,
                                        UserDetailsService userDetailsService, LoginUrlService loginUrlService) {
        super(providers, parent);
        this.userDetailsService = userDetailsService;
        this.loginUrlService = loginUrlService;
    }

    @Override
    public Authentication authenticate(Authentication authentication) throws AuthenticationException {
        try {

            Authentication resultAuthentication = super.authenticate(authentication);

            if (!(authentication instanceof UserAuthenticationHolder)
                    || !(resultAuthentication instanceof UserAuthenticationHolder)) {
                return resultAuthentication;
            }

            UserAuthentication userAuthentication =
                    ((UserAuthenticationHolder) resultAuthentication).getUserAuthentication();
            HttpServletRequest request = ((UserAuthenticationHolder) authentication).getHttpServletRequest();

            return new UserAuthenticationHolder(userDetailsService.addDetails(userAuthentication, request));

        } catch (ProviderNotFoundException e) {
            throw new NoAuthenticationDataException(messages.getMessage(AuthErrorMsg.NO_AUTH_DATA),
                    loginUrlService.getDefaultLoginUrl());
        }
    }
}
