package ru.yandex.partner.libs.auth.model;

import java.util.Collections;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.partner.core.role.Role;
import ru.yandex.partner.libs.auth.RightHolder;
import ru.yandex.partner.libs.rbac.right.Right;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Класс хранит информацию о текущем пользователе, полученную из аутентификации
 */
@ParametersAreNonnullByDefault
public class UserCredentials {
    private final long uid;
    private final String login;
    private final Set<Role> roles;
    private final Set<Right> rights;
    private final Set<String> rightNames;
    private final Set<String> features;

    public UserCredentials(long uid) {
        this(uid, null, Collections.emptySet(), Collections.emptySet(), Collections.emptySet());
    }

    public UserCredentials(long uid,
                           @Nullable String login,
                           Set<Role> roles,
                           Set<Right> rights,
                           Set<String> features) {
        this.uid = uid;
        this.login = login;
        this.roles = checkNotNull(roles);
        this.rights = checkNotNull(rights);
        this.rightNames = rights.stream()
                .map(Right::getName)
                .collect(Collectors.toSet());
        this.features = checkNotNull(features);
    }

    public long getUid() {
        return uid;
    }

    @Nullable
    public String getLogin() {
        return login;
    }

    public Set<Role> getRoles() {
        return roles;
    }

    public Set<Right> getRights() {
        return rights;
    }

    public Set<String> getFeatures() {
        return features;
    }

    public boolean checkRight(String rightName) {
        return rightNames.contains(rightName);
    }

    public boolean checkRight(RightHolder right) {
        return rightNames.contains(right.getRightName());
    }

    public boolean checkShortRight(String accessor, String shortRightName) {
        return checkRight(accessor + "_" + shortRightName);
    }
}
