package ru.yandex.partner.libs.auth.provider

import org.springframework.security.authentication.AuthenticationProvider
import org.springframework.security.core.Authentication
import org.springframework.util.Assert
import ru.yandex.partner.libs.auth.annotation.AuthenticationType
import ru.yandex.partner.libs.auth.annotation.SupportsAuthType
import ru.yandex.partner.libs.auth.model.UserAuthenticationHolder
import javax.servlet.http.HttpServletRequest

class HandlerSupportedAuthTypeProvider(
    private val authenticationProvider: AuthenticationProvider,
) : AuthenticationProvider by authenticationProvider {
    companion object {
        const val SUPPORTED_AUTH_TYPES = "SUPPORTED_AUTH_TYPES"
    }

    private fun checkRequestSupported(request: HttpServletRequest): Boolean {
        if (authenticationProvider !is SupportsAuthType) {
            return true
        }
        // if null or absent - support all auth types, including current
        val supportedAuthTypes = request.getAttribute(SUPPORTED_AUTH_TYPES) ?: return true

        return (supportedAuthTypes as Collection<AuthenticationType>)
            .contains(authenticationProvider.supportedAuthType())
    }

    override fun authenticate(authentication: Authentication?): Authentication? {
        Assert.isInstanceOf(UserAuthenticationHolder::class.java, authentication) {
            "Only UserAuthenticationHolder is supported"
        }

        val request = (authentication as UserAuthenticationHolder).httpServletRequest

        checkNotNull(request)

        return if (checkRequestSupported(request)) authenticationProvider.authenticate(authentication) else null
    }
}
