package ru.yandex.partner.libs.auth.provider;

import javax.servlet.http.HttpServletRequest;

import org.jetbrains.annotations.Nullable;
import org.springframework.security.authentication.AuthenticationProvider;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.AuthenticationException;
import org.springframework.util.Assert;

import ru.yandex.partner.libs.auth.annotation.AuthenticationType;
import ru.yandex.partner.libs.auth.annotation.SupportsAuthType;
import ru.yandex.partner.libs.auth.model.UserAuthentication;
import ru.yandex.partner.libs.auth.model.UserAuthenticationHolder;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Общий класс для провайдеров аутентификации
 */
public abstract class PartnerAuthenticationProvider implements AuthenticationProvider, SupportsAuthType {

    /**
     * Попытка аутентификации
     *
     * @param authentication {@link UserAuthenticationHolder}
     * @return UserAuthenticationHolder с заполненным полем {@link UserAuthentication},
     * либо null если провайдер не может авторизовать данный запрос (нету нужных хедеров)
     * @throws AuthenticationException если провайдер пытается авторизовать, но не успешно (будет конвертировано в 401)
     * @throws RuntimeException        если произошла ошибка сервера (будет конвертировано в 500)
     */
    @Override
    public Authentication authenticate(Authentication authentication) throws AuthenticationException {
        Assert.isInstanceOf(UserAuthenticationHolder.class, authentication,
                () -> "Only UserAuthenticationHolder is supported");

        final HttpServletRequest request = ((UserAuthenticationHolder) authentication).getHttpServletRequest();

        checkNotNull(request);
        if (!checkRequestSupported(request)) {
            return null;
        }

        final UserAuthentication userAuthentication = authenticateHttpServletRequest(request);
        return userAuthentication == null ? null : new UserAuthenticationHolder(userAuthentication);
    }

    /**
     * Наши провайдеры поддерживают только тип {@link UserAuthenticationHolder}
     *
     * @param authentication тип объекта
     * @return нужно ли пытаться авторизоваться
     */
    @Override
    public boolean supports(Class<?> authentication) {
        return (UserAuthenticationHolder.class
                .isAssignableFrom(authentication));
    }

    /**
     * Проверяет, есть ли в запросе нужные хедеры
     *
     * @param request http-запрос
     * @return пытаемся авторизоваться или нет
     */
    public abstract boolean checkRequestSupported(HttpServletRequest request);

    /**
     * Непосредственно производит процесс аутентификации
     *
     * @param request http-запрос
     * @return UserAuthentication
     * @throws AuthenticationException если провайдер пытается авторизовать, но не успешно (будет конвертировано в 401)
     * @throws RuntimeException        если произошла ошибка сервера (будет конвертировано в 500)
     */
    protected abstract UserAuthentication authenticateHttpServletRequest(HttpServletRequest request)
            throws AuthenticationException;

    @Nullable
    @Override
    public AuthenticationType supportedAuthType() {
        return null;
    }
}
