package ru.yandex.partner.libs.auth.provider.apikey;

import java.net.Inet6Address;
import java.net.InetAddress;
import java.net.URI;
import java.net.UnknownHostException;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;

import javax.servlet.http.HttpServletRequest;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.jetbrains.annotations.Nullable;
import org.springframework.http.MediaType;
import org.springframework.security.core.AuthenticationException;
import org.springframework.web.reactive.function.client.WebClient;
import org.springframework.web.util.UriComponentsBuilder;

import ru.yandex.partner.defaultconfiguration.rpc.RpcConfigProps;
import ru.yandex.partner.libs.auth.annotation.AuthenticationType;
import ru.yandex.partner.libs.auth.exception.UserAuthenticationProcessException;
import ru.yandex.partner.libs.auth.exception.authentication.ApiKeyAuthenticationI18nException;
import ru.yandex.partner.libs.auth.message.YandexCabinetMsg;
import ru.yandex.partner.libs.auth.model.AuthenticationMethod;
import ru.yandex.partner.libs.auth.model.UserAuthentication;
import ru.yandex.partner.libs.auth.model.UserCredentials;
import ru.yandex.partner.libs.auth.provider.PartnerAuthenticationProvider;
import ru.yandex.partner.libs.auth.provider.apikey.response.ApiKeyResponse;
import ru.yandex.partner.libs.auth.provider.apikey.response.ApiKeyUser;
import ru.yandex.partner.libs.common.HttpUtils;
import ru.yandex.partner.libs.exceptions.JsonProcessException;

import static com.google.common.base.Preconditions.checkNotNull;
import static ru.yandex.partner.libs.common.PartnerRequestService.retryWithFixedDelay;
import static ru.yandex.partner.libs.common.PartnerRequestService.serviceUnavailable;

/**
 * Провайдер аутентификации для работы с сервисом разработчиков (APIKEYS)
 */
public class ApiKeyAuthenticationProvider extends PartnerAuthenticationProvider {
    private static final String API_KEY_PARAMETER_NAME = "apikey";
    private static final String API_KEY_HEADER_NAME = "Authorization";

    private final ObjectMapper objectMapper;
    private final String apiKeyUrl;
    private final String apiKeyServiceToken;
    private final WebClient webClient;
    private final RpcConfigProps rpcConfigProps;

    public ApiKeyAuthenticationProvider(
            String apiKeyServiceToken,
            WebClient webClient,
            ObjectMapper objectMapper,
            ApiKeyRpcConfig apiKeyRpcConfig
    ) {
        this.apiKeyUrl = checkNotNull(apiKeyRpcConfig.getUrl());
        this.rpcConfigProps = apiKeyRpcConfig;
        this.apiKeyServiceToken = checkNotNull(apiKeyServiceToken);
        this.webClient = webClient;
        this.objectMapper = objectMapper;
    }

    @Override
    protected UserAuthentication authenticateHttpServletRequest(HttpServletRequest request)
            throws AuthenticationException {
        return this.doAuthenticate(HttpUtils.getIpAddrFromRequest(request),
                HttpUtils.getParametersMapFromRequest(request),
                HttpUtils.getHeadersMapFromRequest(request));
    }

    @Override
    public boolean checkRequestSupported(HttpServletRequest request) {
        checkNotNull(request);
        Map<String, List<String>> parameters = HttpUtils.getParametersMapFromRequest(request);
        Map<String, List<String>> headers = HttpUtils.getHeadersMapFromRequest(request);
        if (parameters == null || headers == null) {
            return false;
        }
        return parameters.containsKey(API_KEY_PARAMETER_NAME) || headers.containsKey(API_KEY_HEADER_NAME);
    }

    /**
     * Метод проводит аутентификацию пользователя по параметрам HTTP запроса
     *
     * @param remoteAddr   - адрес пользователя
     * @param parameterMap - хеш массив (имя параметра запроса -> массив значений)
     * @param headerMap    - хеш массив (имя хедера запроса -> массив значений)
     * @return Optional<Authentication> если аутентификация успешная, иначе Optional.empty()
     */
    private UserAuthentication doAuthenticate(String remoteAddr, Map<String, List<String>> parameterMap,
                                              Map<String, List<String>> headerMap) throws AuthenticationException {
        checkNotNull(remoteAddr);

        String key = "";
        if (parameterMap.containsKey(API_KEY_PARAMETER_NAME)) {
            key = parameterMap.get(API_KEY_PARAMETER_NAME) != null
                    && !parameterMap.get(API_KEY_PARAMETER_NAME).isEmpty()
                    ? parameterMap.get(API_KEY_PARAMETER_NAME).get(0) : "";
        } else if (headerMap.containsKey(API_KEY_HEADER_NAME)) {
            key = headerMap.get(API_KEY_HEADER_NAME) != null
                    && !headerMap.get(API_KEY_HEADER_NAME).isEmpty()
                    ? headerMap.get(API_KEY_HEADER_NAME).get(0) : "";
        }

        // Remove optional token mark
        Pattern p = Pattern.compile("^token\\s+");
        key = p.matcher(key).replaceAll("");

        return authenticateKey(remoteAddr, key);
    }

    protected UserAuthentication authenticateKey(String remoteAddr, String key) {
        URI uri = UriComponentsBuilder
                .fromHttpUrl(this.apiKeyUrl)
                .path("/check_key")
                .queryParam("service_token", this.apiKeyServiceToken)
                .queryParam("key", key)
                .queryParam("user_ip", remoteAddr)
                .queryParam("ip_v", ipVersion(remoteAddr))
                .build().toUri();

        String apiKeyResponseString = webClient.get()
                .uri(uri)
                .accept(MediaType.APPLICATION_JSON)
                .exchangeToMono(clientResponse -> clientResponse.toEntity(String.class))
                .transform(retryWithFixedDelay(rpcConfigProps))
                .blockOptional()
                .orElse(serviceUnavailable())
                .getBody();

        if (apiKeyResponseString == null || apiKeyResponseString.isBlank()) {
            throw new UserAuthenticationProcessException("ApiKey /check_key response is empty");
        }
        ApiKeyResponse apiKeyResponse;
        try {
            apiKeyResponse = objectMapper.readValue(apiKeyResponseString, ApiKeyResponse.class);
        } catch (JsonProcessingException e) {
            throw new JsonProcessException("Not valid Json", e);
        }
        if (!apiKeyResponse.isOk()) {
            if (apiKeyResponse.isKeyError()) {
                // ошибка в пользовательском ключе - бросаем 401
                throw new ApiKeyAuthenticationI18nException(YandexCabinetMsg.INVALID_KEY);
            }
            // ошибка на нашей стороне - бросаем 500
            throw new UserAuthenticationProcessException("ApiKey error: " + apiKeyResponse.getError());
        }


        ApiKeyUser apiKeyUser = apiKeyResponse.getKeyInfo().getUser();

        return new UserAuthentication(AuthenticationMethod.AUTH_VIA_API_KEYS_OAUTH,
                new UserCredentials(apiKeyUser.getUid()));
    }

    private String ipVersion(String remoteAddr) {
        try {
            return (InetAddress.getByName(remoteAddr) instanceof Inet6Address) ? "6" : "4";
        } catch (UnknownHostException e) {
            throw new RuntimeException("Unknown host: " + remoteAddr);
        }
    }

    @Nullable
    @Override
    public AuthenticationType supportedAuthType() {
        return AuthenticationType.API_KEY;
    }
}
