package ru.yandex.partner.libs.auth.provider.cookie;

import java.util.Set;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;

import com.sun.jdi.request.InvalidRequestStateException;
import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpStatus;
import org.springframework.security.core.AuthenticationException;
import org.springframework.web.server.ResponseStatusException;

import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxSessionIdException;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxSessionIdException.BlackboxSessionIdStatus;
import ru.yandex.partner.libs.auth.annotation.AuthenticationType;
import ru.yandex.partner.libs.auth.exception.UserAuthenticationProcessException;
import ru.yandex.partner.libs.auth.exception.authentication.CookieAuthenticationI18nException;
import ru.yandex.partner.libs.auth.message.AuthErrorMsg;
import ru.yandex.partner.libs.auth.model.AuthenticationMethod;
import ru.yandex.partner.libs.auth.model.UserAuthentication;
import ru.yandex.partner.libs.auth.model.UserCredentials;
import ru.yandex.partner.libs.auth.provider.PartnerAuthenticationProvider;
import ru.yandex.partner.libs.auth.service.LoginUrlService;
import ru.yandex.partner.libs.common.HttpUtils;
import ru.yandex.partner.libs.extservice.blackbox.BlackboxService;
import ru.yandex.partner.libs.extservice.blackbox.BlackboxUserInfo;

import static com.google.common.base.Preconditions.checkNotNull;

public class CookieAuthenticationProvider extends PartnerAuthenticationProvider {

    private static final Logger LOGGER = LoggerFactory.getLogger(CookieAuthenticationProvider.class);

    private static final String COOKIE_SESSION_ID_NAME = "Session_id";
    private static final String COOKIE_SESSION_ID2_NAME = "sessionid2";
    private static final Set<String> REQUIRED_COOKIE_NAMES = Set.of(COOKIE_SESSION_ID_NAME, COOKIE_SESSION_ID2_NAME);

    private final BlackboxService blackboxService;
    private final LoginUrlService loginUrlService;

    public CookieAuthenticationProvider(BlackboxService blackboxService, LoginUrlService loginUrlService) {
        this.blackboxService = blackboxService;
        this.loginUrlService = loginUrlService;
    }

    @Override
    public boolean checkRequestSupported(HttpServletRequest request) {
        checkNotNull(request);
        int counter = REQUIRED_COOKIE_NAMES.size();
        final Cookie[] cookies = request.getCookies();
        if (cookies != null && cookies.length >= counter) {
            for (Cookie cookie : cookies) {
                if (REQUIRED_COOKIE_NAMES.contains(cookie.getName()) && --counter == 0) {
                    return true;
                }
            }
        }
        return false;
    }

    @Override
    protected UserAuthentication authenticateHttpServletRequest(HttpServletRequest request)
            throws AuthenticationException {
        try {
            return this.doAuthenticate(HttpUtils.getServerHostFromRequest(request), request.getCookies());
        } catch (InvalidRequestStateException e) {
            throw new ResponseStatusException(HttpStatus.BAD_REQUEST, e.getMessage(), e);
        }
    }

    /**
     * Метод проводит аутентификацию пользователя по параметрам HTTP запроса
     *
     * @param host    - хост пользователя
     * @param cookies - массив кук
     * @return Optional<Authentication> если аутентификация успешная, иначе Optional.empty()
     */
    private UserAuthentication doAuthenticate(String host, Cookie[] cookies) {
        String sessionId = getCookieValue(cookies, COOKIE_SESSION_ID_NAME);
        String sslSessionId = getCookieValue(cookies, COOKIE_SESSION_ID2_NAME);

        if (sessionId == null) {
            throw new UserAuthenticationProcessException("SessionId cookie not found");
        }

        if (StringUtils.isEmpty(host)) {
            LOGGER.warn("Could not retrieve host from request");
        }

        try {
            BlackboxUserInfo userInfo = blackboxService.authenticateWithSessionId(sessionId, sslSessionId, host);
            return new UserAuthentication(AuthenticationMethod.AUTH_VIA_COOKIES,
                    new UserCredentials(userInfo.getUid()));
        } catch (BlackboxSessionIdException e) {

            BlackboxSessionIdStatus status = e.getStatus();
            // blackbox забраковал куку - бросаем 401
            throw new CookieAuthenticationI18nException(
                    AuthErrorMsg.forBlackboxResponseStatus(status),
                    loginUrlService.getLoginUrlForBlackboxStatus(status)
            );

        }
    }

    private String getCookieValue(Cookie[] cookies, String cookieName) {
        for (Cookie cookie : cookies) {
            if (cookie.getName().equals(cookieName)) {
                return cookie.getValue();
            }
        }
        return null;
    }

    @Nullable
    @Override
    public AuthenticationType supportedAuthType() {
        return AuthenticationType.COOKIE;
    }
}
