package ru.yandex.partner.libs.auth.provider.tvm;

import java.util.Objects;

import javax.servlet.http.HttpServletRequest;

import org.jetbrains.annotations.Nullable;
import org.springframework.security.core.AuthenticationException;

import ru.yandex.partner.core.CoreConstants;
import ru.yandex.partner.libs.auth.annotation.AuthenticationType;
import ru.yandex.partner.libs.auth.exception.authentication.TvmAuthenticationI18nException;
import ru.yandex.partner.libs.auth.message.AuthErrorMsg;
import ru.yandex.partner.libs.auth.model.AuthenticationMethod;
import ru.yandex.partner.libs.auth.model.TvmUserAuthentication;
import ru.yandex.partner.libs.auth.model.UserAuthentication;
import ru.yandex.partner.libs.auth.model.UserCredentials;
import ru.yandex.partner.libs.auth.provider.PartnerAuthenticationProvider;
import ru.yandex.partner.libs.i18n.MsgWithArgs;
import ru.yandex.partner.libs.tvm.TvmHeaders;
import ru.yandex.passport.tvmauth.CheckedServiceTicket;
import ru.yandex.passport.tvmauth.CheckedUserTicket;
import ru.yandex.passport.tvmauth.TvmClient;

import static com.google.common.base.Preconditions.checkNotNull;

public class TvmAuthenticationProvider extends PartnerAuthenticationProvider {
    private final TvmClient tvmClient;
    private final int selfServiceId;

    public TvmAuthenticationProvider(TvmClient tvmClient, int selfServiceId) {
        this.tvmClient = tvmClient;
        this.selfServiceId = selfServiceId;
    }

    @Override
    public boolean checkRequestSupported(HttpServletRequest request) {
        checkNotNull(request);
        return request.getHeader(TvmHeaders.SERVICE_TICKET_HEADER_NAME) != null;
    }

    @Override
    protected UserAuthentication authenticateHttpServletRequest(HttpServletRequest request)
            throws AuthenticationException {
        String serviceTicketString =
                Objects.requireNonNullElse(request.getHeader(TvmHeaders.SERVICE_TICKET_HEADER_NAME), "");
        String userTicketString = request.getHeader(TvmHeaders.USER_TICKET_HEADER_NAME);
        String userIdString = request.getHeader(TvmHeaders.USER_ID_HEADER);

        CheckedServiceTicket serviceTicket = tvmClient.checkServiceTicket(serviceTicketString);
        if (!serviceTicket.booleanValue()) {
            throw new TvmAuthenticationI18nException(
                    MsgWithArgs.of(AuthErrorMsg.SERVICE_TICKET_STATUS, serviceTicket.getStatus()
                    ));
        }

        boolean selfAuth = selfServiceId == serviceTicket.getSrc();
        if (userTicketString != null) {
            return authUserTicket(selfAuth, userTicketString);
        } else if (userIdString != null) {
            if (selfAuth) {
                return authUserIdHeader(userIdString);
            } else {
                throw new TvmAuthenticationI18nException(
                        MsgWithArgs.of(AuthErrorMsg.FORBIDDEN_HEADERS, TvmHeaders.USER_ID_HEADER));
            }
        } else {
            return new TvmUserAuthentication(
                    selfAuth,
                    AuthenticationMethod.AUTH_VIA_TVM_SERVICE,
                    new UserCredentials(CoreConstants.INTAPI_USER_ID)); // todo: actual service ID
        }
    }

    private TvmUserAuthentication authUserTicket(boolean selfAuth, String userTicketString) {
        CheckedUserTicket userTicket = tvmClient.checkUserTicket(userTicketString);
        if (!userTicket.booleanValue()) {
            throw new TvmAuthenticationI18nException(
                    MsgWithArgs.of(AuthErrorMsg.USER_TICKET_STATUS, userTicket.getStatus())
            );
        }
        return new TvmUserAuthentication(
                selfAuth,
                AuthenticationMethod.AUTH_VIA_TVM_USER,
                new UserCredentials(userTicket.getDefaultUid()));
    }

    private TvmUserAuthentication authUserIdHeader(String realUserIdString) {
        try {
            long userId = Long.parseLong(realUserIdString);

            return new TvmUserAuthentication(
                    true,
                    AuthenticationMethod.AUTH_VIA_TVM_SERVICE_AND_HEADER,
                    new UserCredentials(userId)
            );
        } catch (NumberFormatException e) {
            throw new TvmAuthenticationI18nException(
                    MsgWithArgs.of(AuthErrorMsg.HEADER_MUST_BE_INTEGER_NUMBER, TvmHeaders.USER_ID_HEADER));
        }
    }

    @Nullable
    @Override
    public AuthenticationType supportedAuthType() {
        return AuthenticationType.TVM;
    }
}
