package ru.yandex.partner.libs.auth.service;

import java.util.Arrays;
import java.util.Set;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;

import org.springframework.core.env.Environment;
import org.springframework.stereotype.Service;

import ru.yandex.partner.core.entity.QueryOpts;
import ru.yandex.partner.core.entity.user.filter.UserFilters;
import ru.yandex.partner.core.entity.user.model.CommonUser;
import ru.yandex.partner.core.entity.user.model.User;
import ru.yandex.partner.core.entity.user.service.UserService;
import ru.yandex.partner.core.exceptions.UserNotFoundByLoginException;
import ru.yandex.partner.core.filter.CoreFilterNode;
import ru.yandex.partner.libs.auth.exception.response.FakeLoginBadRequestException;
import ru.yandex.partner.libs.auth.exception.response.FakeLoginForbiddenException;
import ru.yandex.partner.libs.auth.message.RestApiControllerMsg;
import ru.yandex.partner.libs.auth.message.WebInterfaceMsg;
import ru.yandex.partner.libs.auth.model.AuthenticationMethod;
import ru.yandex.partner.libs.auth.model.FakeUserAuthentication;
import ru.yandex.partner.libs.auth.model.UserAuthentication;
import ru.yandex.partner.libs.auth.model.UserCredentials;
import ru.yandex.partner.libs.rbac.constants.Constants;

import static com.google.common.base.Preconditions.checkNotNull;
import static ru.yandex.partner.libs.rbac.constants.Constants.FAKE_LOGIN_RIGHT_NAME;
import static ru.yandex.partner.libs.rbac.constants.Constants.PRODUCTION;

@Service
public class FakeLoginService {
    private final UserService userService;
    private final Environment environment;

    public FakeLoginService(UserService userService, Environment environment) {
        this.userService = checkNotNull(userService);
        this.environment = checkNotNull(environment);
    }

    public FakeUserAuthentication tryFakeLogin(UserAuthentication userAuthentication, HttpServletRequest request) {
        if (request.getCookies() == null) {
            return null;
        }
        String fakeLogin = null;
        for (Cookie cookie : request.getCookies()) {
            if (Constants.FAKE_LOGIN_COOKIE_NAME.equals(cookie.getName())) {
                fakeLogin = cookie.getValue();
                break;
            }
        }

        if (fakeLogin == null) {
            return null;
        }

        checkEnvironment();
        checkRoleRight(userAuthentication);
        try {
            CommonUser commonUser = userService.findAll(QueryOpts.forClass(User.class)
                    .withFilter(CoreFilterNode.eq(UserFilters.LOGIN, fakeLogin))
                    .withProps(Set.of(User.ID))
            ).stream().findFirst().orElseThrow(UserNotFoundByLoginException::new);
            return new FakeUserAuthentication(userAuthentication, AuthenticationMethod.AUTH_VIA_COOKIES,
                    new UserCredentials(commonUser.getId()));
        } catch (UserNotFoundByLoginException e) {
            throw new FakeLoginBadRequestException(WebInterfaceMsg.BAD_FAKE_LOGIN, e);
        }
    }

    private void checkEnvironment() {
        if (Arrays.stream(environment.getActiveProfiles())
                .anyMatch(profile -> profile.equalsIgnoreCase(PRODUCTION))) {
            throw new FakeLoginForbiddenException(RestApiControllerMsg.FAKE_LOGIN_NOT_ALLOWED);
        }
    }

    private void checkRoleRight(UserAuthentication userAuthentication) {
        if (!userAuthentication.userHasRight(FAKE_LOGIN_RIGHT_NAME)) {
            throw new FakeLoginForbiddenException(RestApiControllerMsg.FAKE_LOGIN_ACCESS_DENIED);
        }
    }
}

