package ru.yandex.partner.libs.auth.service;

import java.util.HashSet;
import java.util.Set;

import javax.servlet.http.HttpServletRequest;

import com.google.common.collect.Sets;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import ru.yandex.direct.model.ModelProperty;
import ru.yandex.partner.core.CoreConstants;
import ru.yandex.partner.core.entity.QueryOpts;
import ru.yandex.partner.core.entity.user.filter.UserFilters;
import ru.yandex.partner.core.entity.user.model.User;
import ru.yandex.partner.core.entity.user.service.UserAdfoxService;
import ru.yandex.partner.core.entity.user.service.UserService;
import ru.yandex.partner.core.exceptions.UserNotFoundException;
import ru.yandex.partner.core.filter.CoreFilterNode;
import ru.yandex.partner.core.role.Role;
import ru.yandex.partner.libs.auth.exception.UserAuthenticationProcessException;
import ru.yandex.partner.libs.auth.exception.authentication.NotRegisteredAuthenticationException;
import ru.yandex.partner.libs.auth.model.FakeUserAuthentication;
import ru.yandex.partner.libs.auth.model.UserAuthentication;
import ru.yandex.partner.libs.auth.model.UserCredentials;
import ru.yandex.partner.libs.extservice.blackbox.BlackboxService;
import ru.yandex.partner.libs.extservice.blackbox.BlackboxUserInfo;
import ru.yandex.partner.libs.rbac.right.Right;
import ru.yandex.partner.libs.rbac.role.RoleSet;
import ru.yandex.partner.libs.rbac.userrole.UserRoleService;

import static com.google.common.base.Preconditions.checkNotNull;
import static ru.yandex.partner.core.entity.user.model.prop.BaseUserIdPropHolder.ID;
import static ru.yandex.partner.core.entity.user.model.prop.BaseUserUidPropHolder.UID;
import static ru.yandex.partner.core.entity.user.model.prop.CommonUserLoginPropHolder.LOGIN;
import static ru.yandex.partner.core.entity.user.model.prop.UserWithFeaturesFeaturesPropHolder.FEATURES;
import static ru.yandex.partner.core.entity.user.model.prop.UserWithRolesRolesPropHolder.ROLES;

/**
 * Сервис, насыщающий объект {@link UserAuthentication} данными
 * 1) Fake login если есть
 * 2) Данные пользователя {@link User}
 * 3) Права {@link Right}
 */
@Service
public class UserDetailsService {
    public static final String CAN_VIEW_ADFOX_FIELDS = "can_view_adfox_fields";
    public static final Set<ModelProperty<?, ?>> USER_AUTH_PROPERTIES =
            Set.of(ID, UID, LOGIN, ROLES, FEATURES);
    public static final Set<ModelProperty<?, ?>> SYSTEM_CRON_AUTH_PROPERTIES =
            Set.of(ID, UID, LOGIN);

    private static final Logger LOGGER = LoggerFactory.getLogger(UserDetailsService.class);

    private final UserService userService;
    private final UserAdfoxService userAdfoxService;
    private final UserRoleService userRoleService;
    private final FakeLoginService fakeLoginService;
    private final BlackboxService blackboxService;

    public UserDetailsService(UserService userService,
                              UserAdfoxService userAdfoxService,
                              UserRoleService userRoleService,
                              FakeLoginService fakeLoginService,
                              BlackboxService blackboxService) {
        this.userService = userService;
        this.userAdfoxService = userAdfoxService;
        this.userRoleService = userRoleService;
        this.fakeLoginService = fakeLoginService;
        this.blackboxService = blackboxService;
    }

    /**
     * @throws UserAuthenticationProcessException ошибка в процессе аутентификации
     */
    public UserAuthentication addDetails(UserAuthentication userAuthentication, HttpServletRequest httpServletRequest) {
        checkNotNull(httpServletRequest);
        checkNotNull(userAuthentication);

        try {
            UserAuthentication realUserAuthentication = loadDetails(userAuthentication);

            FakeUserAuthentication fakeUserAuthentication =
                    fakeLoginService.tryFakeLogin(realUserAuthentication, httpServletRequest);

            if (fakeUserAuthentication != null) {
                return loadDetails(fakeUserAuthentication);
            } else {
                return realUserAuthentication;
            }
        } catch (UserNotFoundException e) {
            throw new NotRegisteredAuthenticationException("User not found. Uid = " + userAuthentication.getUid());
        }
    }

    private UserAuthentication loadDetails(UserAuthentication userAuthentication) {
        long uid = userAuthentication.getUid();

        if (uid == CoreConstants.SYSTEM_CRON_USER_ID) {
            return loadSystemCronUser(userAuthentication);
        } else {
            return loadSimpleUser(userAuthentication);
        }
    }

    private UserAuthentication loadSimpleUser(UserAuthentication userAuthentication) {
        long uid = userAuthentication.getUid();

        BlackboxUserInfo blackboxUserInfo = loadUserInfo(uid);
        User user = userService.findAll(QueryOpts.forClass(User.class)
                .withFilter(CoreFilterNode.eq(UserFilters.UID, uid))
                .withProps(USER_AUTH_PROPERTIES)
        ).stream().findFirst().orElseThrow(UserNotFoundException::new);
        Set<Role> roles = user.getRoles();
        Set<Right> rolesRights = userRoleService.getUserRightsFromRoles(roles);
        Set<Right> rights = calculateRights(uid, roles, rolesRights);
        Set<String> features = user.getFeatures() == null
                ? Set.of()
                : new HashSet<>(user.getFeatures());

        UserAuthentication reachUserAuthentication = new UserAuthentication(
                userAuthentication.getAuthenticationMethod(),
                new UserCredentials(user.getUid(), user.getLogin(), roles, rights, features),
                userAuthentication.isSelfAuth());
        reachUserAuthentication.setInternalUser(user);
        reachUserAuthentication.setBlackboxUserInfo(blackboxUserInfo);

        return reachUserAuthentication;
    }

    private UserAuthentication loadSystemCronUser(UserAuthentication userAuthentication) {
        long uid = userAuthentication.getUid();
        User user = userService.findAll(QueryOpts.forClass(User.class)
                .withFilter(CoreFilterNode.eq(UserFilters.UID, uid))
                .withProps(SYSTEM_CRON_AUTH_PROPERTIES)
        ).get(0);

        Set<Right> rolesRights = userRoleService.getUserRightsFromRoles(RoleSet.getRoles());
        Set<Right> rights = Sets.newHashSetWithExpectedSize(rolesRights.size() + 1);
        rights.addAll(rolesRights);
        rights.add(new Right(CAN_VIEW_ADFOX_FIELDS, Set.of()));

        UserAuthentication reachUserAuthentication = new UserAuthentication(
                userAuthentication.getAuthenticationMethod(),
                new UserCredentials(user.getUid(), user.getLogin(), Set.of(), rights, Set.of()),
                userAuthentication.isSelfAuth()
        );

        reachUserAuthentication.setInternalUser(user);
        reachUserAuthentication.setBlackboxUserInfo(new BlackboxUserInfo());

        return reachUserAuthentication;
    }

    private BlackboxUserInfo loadUserInfo(long uid) {
        try {
            return blackboxService.getUserInfo(uid);
        } catch (Exception e) {
            LOGGER.error("Error while requesting user language from blackbox. Uid: {}", uid, e);
        }
        return new BlackboxUserInfo();
    }

    private Set<Right> calculateRights(long userId, Set<Role> roles, Set<Right> curRights) {
        boolean hasAdfox = userAdfoxService.hasAdfoxAccount(userId);

        var rights = Sets.<Right>newHashSetWithExpectedSize(curRights.size() + 1);
        rights.addAll(curRights);
        if (hasAdfox || roles.stream().anyMatch(Role::isInternal)) {
            rights.add(new Right(CAN_VIEW_ADFOX_FIELDS, Set.of()));
        }

        return rights;
    }
}
