package ru.yandex.partner.libs.auth.model;

import java.util.HashSet;
import java.util.Set;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

import ru.yandex.partner.core.entity.user.model.User;
import ru.yandex.partner.core.role.Role;
import ru.yandex.partner.libs.rbac.right.Right;
import ru.yandex.partner.libs.rbac.role.RoleSet;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertNull;
import static org.junit.jupiter.api.Assertions.assertTrue;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;
import static ru.yandex.partner.libs.rbac.constants.Constants.FAKE_LOGIN_RIGHT_NAME;

class UserAuthenticationTest {
    UserAuthentication userAuthentication;

    @BeforeEach
    public void initialize() {
        UserCredentials userCredentials = new UserCredentials(
                1L, "login",
                new HashSet<>(),
                new HashSet<>(),
                new HashSet<>());
        userAuthentication = new UserAuthentication(AuthenticationMethod.AUTH_VIA_COOKIES, userCredentials);
    }


    @Test
    void getAuthenticationMethod() {
        assertEquals(AuthenticationMethod.AUTH_VIA_COOKIES, userAuthentication.getAuthenticationMethod());
        assertEquals(5, AuthenticationMethod.values().length);
    }

    @Test
    void getUid() {
        assertEquals(1, userAuthentication.getUid());
        assertEquals("login", userAuthentication.getLogin());
        var fakeUserAuthentication = new FakeUserAuthentication(
                userAuthentication,
                userAuthentication.getAuthenticationMethod(),
                new UserCredentials(
                        2L, "login2",
                        new HashSet<>(),
                        new HashSet<>(),
                        new HashSet<>())
        );
        assertEquals(2, fakeUserAuthentication.getUid());
        assertEquals("login2", fakeUserAuthentication.getLogin());
    }

    @Test
    void getFakeAndRealData() {
        var fakeUserAuthentication = new FakeUserAuthentication(
                userAuthentication,
                userAuthentication.getAuthenticationMethod(),
                new UserCredentials(
                        2L, "login2",
                        new HashSet<>(),
                        new HashSet<>(),
                        new HashSet<>())
        );

        UserCredentials current = fakeUserAuthentication.getUserCredentials();
        UserCredentials real = fakeUserAuthentication.getRealAuthentication().getUserCredentials();

        assertEquals(2, current.getUid());
        assertEquals(1, real.getUid());

        assertEquals("login2", current.getLogin());
        assertEquals("login", real.getLogin());
    }

    @Test
    void getRoles() {
        final Set<Role> developerRolesSet = Set.of(RoleSet.DEVELOPER);

        assertEquals(0, userAuthentication.getRoles().size());

        final UserCredentials mock = mock(UserCredentials.class);
        var fakeUserAuthentication = new FakeUserAuthentication(
                userAuthentication,
                userAuthentication.getAuthenticationMethod(),
                mock
        );
        when(mock.getRoles()).thenReturn(developerRolesSet);

        assertEquals(1, fakeUserAuthentication.getRoles().size());
        assertTrue(fakeUserAuthentication.getRoles().contains(RoleSet.DEVELOPER));
    }

    @Test
    void getRights() {
        final Set<Role> roles = Set.of(RoleSet.DEVELOPER);
        final Right right = new Right(FAKE_LOGIN_RIGHT_NAME, roles);
        final Set<Right> rights = Set.of(right);

        assertEquals(0, userAuthentication.getUserCredentials().getRights().size());
        var fakeUserAuthentication = new FakeUserAuthentication(
                userAuthentication,
                userAuthentication.getAuthenticationMethod(),
                new UserCredentials(
                        2L, "login2",
                        roles,
                        rights,
                        Set.of())
        );

        assertEquals(1, fakeUserAuthentication.getUserCredentials().getRights().size());
        assertTrue(fakeUserAuthentication.getUserCredentials().getRights().contains(right));

    }

    @Test
    void userTest() {
        assertNull(userAuthentication.getInternalUser());

        final User mock = mock(User.class);
        userAuthentication.setInternalUser(mock);

        assertEquals(mock, userAuthentication.getInternalUser());
    }
}
