package ru.yandex.partner.libs.auth.provider.apikey;

import java.io.IOException;
import java.net.URI;
import java.nio.charset.StandardCharsets;

import javax.servlet.http.HttpServletRequest;

import com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.commons.io.IOUtils;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.mockito.Mockito;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.mock.web.MockHttpServletRequest;
import org.springframework.security.core.Authentication;
import org.springframework.web.reactive.function.client.ClientResponse;
import org.springframework.web.reactive.function.client.ExchangeFunction;
import org.springframework.web.reactive.function.client.WebClient;
import org.springframework.web.util.UriComponentsBuilder;
import reactor.core.publisher.Mono;

import ru.yandex.partner.libs.auth.exception.UserAuthenticationProcessException;
import ru.yandex.partner.libs.auth.exception.authentication.AuthenticationI18nException;
import ru.yandex.partner.libs.auth.message.YandexCabinetMsg;
import ru.yandex.partner.libs.auth.model.AuthenticationMethod;
import ru.yandex.partner.libs.auth.model.UserAuthentication;
import ru.yandex.partner.libs.auth.model.UserAuthenticationHolder;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertFalse;
import static org.junit.jupiter.api.Assertions.assertNull;
import static org.junit.jupiter.api.Assertions.assertThrows;
import static org.junit.jupiter.api.Assertions.assertTrue;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.doReturn;
import static org.mockito.Mockito.mock;

public class ApiKeyAuthenticationProviderTest {

    private static final String MOCK_APIKEY_URL = "http://mock.url";
    private static final String MOCK_SERVICE_TOKEN = "mock:service:token";

    private static final String FAKE_KEY = "fake-key";
    private static final String USER_IP = "1.1.1.1";

    private ApiKeyAuthenticationProvider apiKeyAuthenticationProvider;

    private WebClient webClient;
    private ExchangeFunction web;
    private ObjectMapper objectMapper;

    @BeforeEach
    void init() {
        web = mock(ExchangeFunction.class);
        webClient = WebClient.builder().exchangeFunction(web).build();
        objectMapper = new ObjectMapper();
        var fakeConf = new FakeApiKeyConfig(MOCK_APIKEY_URL);
        apiKeyAuthenticationProvider = new ApiKeyAuthenticationProvider(
                MOCK_SERVICE_TOKEN,
                webClient,
                objectMapper,
                fakeConf
        );
    }

    @Test
    void testSuccessHeader() throws IOException {
        MockHttpServletRequest request = new MockHttpServletRequest();
        request.addHeader("X-Real-IP", USER_IP);
        request.addHeader("Authorization", FAKE_KEY);
        baseTestSuccess(request);
    }

    @Test
    void testSuccessHeaderWithTokenMark() throws IOException {
        MockHttpServletRequest request = new MockHttpServletRequest();
        request.addHeader("X-Real-IP", USER_IP);
        request.addHeader("Authorization", "token " + FAKE_KEY);
        baseTestSuccess(request);
    }

    @Test
    void testSuccessParameter() throws IOException {
        MockHttpServletRequest request = new MockHttpServletRequest();
        request.addHeader("X-Real-IP", USER_IP);
        request.addParameter("apikey", FAKE_KEY);
        baseTestSuccess(request);
    }

    @Test
    void testNotSupported() {
        MockHttpServletRequest request = new MockHttpServletRequest();
        UserAuthenticationHolder authenticationInput = new UserAuthenticationHolder(request);
        Authentication authenticationOutput = apiKeyAuthenticationProvider.authenticate(authenticationInput);
        assertNull(authenticationOutput);
    }

    @Test
    void testNotAuthenticated() {
        MockHttpServletRequest request = new MockHttpServletRequest();
        request.addHeader("X-Real-IP", USER_IP);
        request.addParameter("apikey", FAKE_KEY);

        doReturn(Mono.just(ClientResponse.create(HttpStatus.NOT_FOUND)
                .header(HttpHeaders.CONTENT_TYPE, "text/json")
                .body("{\"error\": \"Key not found\"}")
                .build()))
                .when(web).exchange(any());

        UserAuthenticationHolder authenticationInput = new UserAuthenticationHolder(request);
        AuthenticationI18nException exception = assertThrows(AuthenticationI18nException.class,
                () -> apiKeyAuthenticationProvider.authenticate(authenticationInput));

        assertEquals(YandexCabinetMsg.INVALID_KEY, exception.getI18nMessage());
    }

    @Test
    void testWrongServiceKey() {
        MockHttpServletRequest request = new MockHttpServletRequest();
        request.addHeader("X-Real-IP", USER_IP);
        request.addParameter("apikey", FAKE_KEY);

        doReturn(Mono.just(ClientResponse.create(HttpStatus.NOT_FOUND)
                .header(HttpHeaders.CONTENT_TYPE, "text/json")
                .body("{\"error\": \"Service not found\"}")
                .build())).when(web).exchange(any());

        UserAuthenticationHolder authenticationInput = new UserAuthenticationHolder(request);
        Exception exception = assertThrows(UserAuthenticationProcessException.class,
                () -> apiKeyAuthenticationProvider.authenticate(authenticationInput));

        assertTrue(exception.getMessage().contains("Service not found"));
    }


    void baseTestSuccess(HttpServletRequest request) throws IOException {

        int userID = 285463599;

        URI uriToCall = UriComponentsBuilder.fromUriString(
                String.format("%s/check_key?service_token=%s&key=%s&user_ip=%s&ip_v=4",
                        MOCK_APIKEY_URL, MOCK_SERVICE_TOKEN, FAKE_KEY, USER_IP
                )).build().toUri();

        String apikeyResponse = IOUtils.toString(
                this.getClass().getClassLoader().getResourceAsStream("apikey-response.json"),
                "UTF-8"
        );

        doReturn(Mono.just(ClientResponse.create(HttpStatus.OK)
                // content type broken intentionally
                .headers(headers -> headers.setContentType(new MediaType(MediaType.TEXT_HTML, StandardCharsets.UTF_8)))
                .body(apikeyResponse)
                .build()
        )).when(web)
                .exchange(Mockito.argThat(req -> req.url().equals(uriToCall)));

        UserAuthenticationHolder authenticationInput = new UserAuthenticationHolder(request);
        Authentication authenticationOutput = apiKeyAuthenticationProvider.authenticate(authenticationInput);

        assertEquals(UserAuthenticationHolder.class, authenticationOutput.getClass());

        UserAuthentication authentication = (UserAuthentication) authenticationOutput.getDetails();

        assertEquals(userID, authentication.getUid());
        assertEquals(AuthenticationMethod.AUTH_VIA_API_KEYS_OAUTH, authentication.getAuthenticationMethod());
        assertFalse(authentication.isSelfAuth());
    }

}

