package ru.yandex.partner.libs.auth.provider.cookie;

import javax.servlet.http.Cookie;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.springframework.mock.web.MockHttpServletRequest;
import org.springframework.security.core.Authentication;

import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxSessionIdException;
import ru.yandex.partner.libs.auth.exception.authentication.AuthenticationI18nException;
import ru.yandex.partner.libs.auth.message.AuthErrorMsg;
import ru.yandex.partner.libs.auth.model.AuthenticationMethod;
import ru.yandex.partner.libs.auth.model.UserAuthentication;
import ru.yandex.partner.libs.auth.model.UserAuthenticationHolder;
import ru.yandex.partner.libs.auth.service.LoginUrlService;
import ru.yandex.partner.libs.extservice.blackbox.BlackboxService;
import ru.yandex.partner.libs.extservice.blackbox.BlackboxUserInfo;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertFalse;
import static org.junit.jupiter.api.Assertions.assertNull;
import static org.junit.jupiter.api.Assertions.assertThrows;
import static org.mockito.Mockito.doReturn;
import static org.mockito.Mockito.doThrow;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.spy;

public class CookieAuthenticationProviderTest {

    public static final String SESSION_ID_VALUE = "session_1_value";
    public static final String SESSIONID2_VALUE = "session_2_value";

    private static final String USER_IP = "1.1.1.1";
    private static final String SERVER_HOST = "my.host";
    public static final String LANGUAGE = "ru";
    public static final long USER_ID = 274;

    private CookieAuthenticationProvider cookieAuthenticationProvider;

    private BlackboxService blackboxService;
    private LoginUrlService loginUrlService;

    @BeforeEach
    void init() {
        blackboxService = mock(BlackboxService.class);
        loginUrlService = spy(new LoginUrlService("test"));

        cookieAuthenticationProvider = new CookieAuthenticationProvider(blackboxService, loginUrlService);
    }

    @Test
    void testSuccess() {

        MockHttpServletRequest request = new MockHttpServletRequest();
        request.setCookies(new Cookie("Session_id", SESSION_ID_VALUE), new Cookie("sessionid2", SESSIONID2_VALUE));
        request.addHeader("X-Real-IP", USER_IP);
        request.addHeader("Host", SERVER_HOST);

        BlackboxUserInfo blackboxUserInfo = new BlackboxUserInfo(USER_ID, LANGUAGE, null);
        doReturn(blackboxUserInfo).when(blackboxService)
                .authenticateWithSessionId(SESSION_ID_VALUE, SESSIONID2_VALUE, SERVER_HOST);

        UserAuthenticationHolder authenticationInput = new UserAuthenticationHolder(request);
        Authentication authenticationOutput = cookieAuthenticationProvider.authenticate(authenticationInput);

        assertEquals(UserAuthenticationHolder.class, authenticationOutput.getClass());

        UserAuthentication authentication = (UserAuthentication) authenticationOutput.getDetails();

        assertEquals(USER_ID, authentication.getUid());
        assertEquals(AuthenticationMethod.AUTH_VIA_COOKIES, authentication.getAuthenticationMethod());
        assertFalse(authentication.isSelfAuth());
    }

    @Test
    void testNotSupportedNoCookie() {
        MockHttpServletRequest request = new MockHttpServletRequest();
        basicTestNotSupported(request);
    }

    @Test
    void testNotSupportedFirstCookie() {
        MockHttpServletRequest request = new MockHttpServletRequest();
        request.setCookies(new Cookie("Session_id", SESSION_ID_VALUE));
        basicTestNotSupported(request);
    }

    @Test
    void testNotSupportedSecondCookie() {
        MockHttpServletRequest request = new MockHttpServletRequest();
        request.setCookies(new Cookie("sessionid2", SESSIONID2_VALUE));
        basicTestNotSupported(request);
    }

    @Test
    void testNotAuthenticated() {
        MockHttpServletRequest request = new MockHttpServletRequest();
        request.setCookies(new Cookie("Session_id", SESSION_ID_VALUE), new Cookie("sessionid2", SESSIONID2_VALUE));
        request.addHeader("X-Real-IP", USER_IP);
        request.addHeader("Host", SERVER_HOST);

        BlackboxSessionIdException ex =
                new BlackboxSessionIdException(BlackboxSessionIdException.BlackboxSessionIdStatus.EXPIRED, "", "");

        doThrow(ex).when(blackboxService)
                .authenticateWithSessionId(SESSION_ID_VALUE, SESSIONID2_VALUE, SERVER_HOST);

        UserAuthenticationHolder authenticationInput = new UserAuthenticationHolder(request);
        AuthenticationI18nException exception = assertThrows(AuthenticationI18nException.class,
                () -> cookieAuthenticationProvider.authenticate(authenticationInput));

        assertEquals(AuthErrorMsg.AUTH_COOKIE_STATUS_EXPIRED, exception.getI18nMessage());
    }

    private void basicTestNotSupported(MockHttpServletRequest request) {
        UserAuthenticationHolder authenticationInput = new UserAuthenticationHolder(request);
        Authentication authenticationOutput = cookieAuthenticationProvider.authenticate(authenticationInput);
        assertNull(authenticationOutput);
    }


}
