package ru.yandex.partner.libs.auth.provider.tvm;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.springframework.mock.web.MockHttpServletRequest;
import org.springframework.security.core.Authentication;

import ru.yandex.partner.libs.auth.exception.authentication.AuthenticationI18nException;
import ru.yandex.partner.libs.auth.message.AuthErrorMsg;
import ru.yandex.partner.libs.auth.model.AuthenticationMethod;
import ru.yandex.partner.libs.auth.model.UserAuthentication;
import ru.yandex.partner.libs.auth.model.UserAuthenticationHolder;
import ru.yandex.partner.libs.i18n.MsgWithArgs;
import ru.yandex.passport.tvmauth.CheckedServiceTicket;
import ru.yandex.passport.tvmauth.CheckedUserTicket;
import ru.yandex.passport.tvmauth.TicketStatus;
import ru.yandex.passport.tvmauth.TvmClient;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertFalse;
import static org.junit.jupiter.api.Assertions.assertNull;
import static org.junit.jupiter.api.Assertions.assertThrows;
import static org.junit.jupiter.api.Assertions.assertTrue;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;
import static ru.yandex.partner.libs.tvm.TvmHeaders.SERVICE_TICKET_HEADER_NAME;
import static ru.yandex.partner.libs.tvm.TvmHeaders.USER_ID_HEADER;
import static ru.yandex.partner.libs.tvm.TvmHeaders.USER_TICKET_HEADER_NAME;

public class TvmAuthenticationProviderTest {

    private static final String SERVICE_TOKEN = "some:service:token";
    private static final String USER_TOKEN = "some:user:token";

    private TvmAuthenticationProvider tvmAuthenticationProvider;

    private TvmClient tvmClient;
    private int selfId;


    @BeforeEach
    void init() {
        tvmClient = mock(TvmClient.class);
        selfId = (int) (Math.random() * 100_000);
        tvmAuthenticationProvider = new TvmAuthenticationProvider(tvmClient, selfId);
    }

    @Test
    void testSuccessService() {

        MockHttpServletRequest request = new MockHttpServletRequest();
        request.addHeader(SERVICE_TICKET_HEADER_NAME, SERVICE_TOKEN);

        CheckedServiceTicket serviceTicket = mock(CheckedServiceTicket.class);
        when(serviceTicket.booleanValue()).thenReturn(true);
        when(tvmClient.checkServiceTicket(SERVICE_TOKEN)).thenReturn(serviceTicket);

        UserAuthenticationHolder authenticationInput = new UserAuthenticationHolder(request);
        Authentication authenticationOutput = tvmAuthenticationProvider.authenticate(authenticationInput);

        assertEquals(UserAuthenticationHolder.class, authenticationOutput.getClass());

        UserAuthentication authentication = (UserAuthentication) authenticationOutput.getDetails();

        assertEquals(337625302L, authentication.getUid());
        assertEquals(AuthenticationMethod.AUTH_VIA_TVM_SERVICE, authentication.getAuthenticationMethod());
        assertFalse(authentication.isSelfAuth());

    }

    @Test
    void testSuccessUser() {

        MockHttpServletRequest request = new MockHttpServletRequest();
        request.addHeader(SERVICE_TICKET_HEADER_NAME, SERVICE_TOKEN);
        request.addHeader(USER_TICKET_HEADER_NAME, USER_TOKEN);

        long userID = 274;

        CheckedServiceTicket serviceTicket = mock(CheckedServiceTicket.class);
        when(serviceTicket.booleanValue()).thenReturn(true);
        when(tvmClient.checkServiceTicket(SERVICE_TOKEN)).thenReturn(serviceTicket);

        CheckedUserTicket userTicket = mock(CheckedUserTicket.class);
        when(userTicket.booleanValue()).thenReturn(true);
        when(userTicket.getDefaultUid()).thenReturn(userID);
        when(tvmClient.checkUserTicket(USER_TOKEN)).thenReturn(userTicket);

        UserAuthenticationHolder authenticationInput = new UserAuthenticationHolder(request);
        Authentication authenticationOutput = tvmAuthenticationProvider.authenticate(authenticationInput);

        assertEquals(UserAuthenticationHolder.class, authenticationOutput.getClass());

        UserAuthentication authentication = (UserAuthentication) authenticationOutput.getDetails();

        assertEquals(userID, authentication.getUid());
        assertEquals(AuthenticationMethod.AUTH_VIA_TVM_USER, authentication.getAuthenticationMethod());
        assertFalse(authentication.isSelfAuth());

    }

    @Test
    void testNotSupported() {
        MockHttpServletRequest request = new MockHttpServletRequest();
        UserAuthenticationHolder authenticationInput = new UserAuthenticationHolder(request);
        Authentication authenticationOutput = tvmAuthenticationProvider.authenticate(authenticationInput);
        assertNull(authenticationOutput);
    }

    @Test
    void testNotAuthenticatedService() {

        MockHttpServletRequest request = new MockHttpServletRequest();
        request.addHeader(SERVICE_TICKET_HEADER_NAME, SERVICE_TOKEN);
        request.addHeader(USER_TICKET_HEADER_NAME, USER_TOKEN);

        CheckedServiceTicket serviceTicket = mock(CheckedServiceTicket.class);
        when(serviceTicket.getStatus()).thenReturn(TicketStatus.EXPIRED);
        when(serviceTicket.booleanValue()).thenReturn(false);
        when(tvmClient.checkServiceTicket(SERVICE_TOKEN)).thenReturn(serviceTicket);

        UserAuthenticationHolder authenticationInput = new UserAuthenticationHolder(request);
        AuthenticationI18nException exception = assertThrows(AuthenticationI18nException.class,
                () -> tvmAuthenticationProvider.authenticate(authenticationInput));

        assertEquals(MsgWithArgs.of(AuthErrorMsg.SERVICE_TICKET_STATUS, TicketStatus.EXPIRED),
                exception.getI18nMessage());
    }

    @Test
    void testNotAuthenticatedUser() {

        MockHttpServletRequest request = new MockHttpServletRequest();
        request.addHeader(SERVICE_TICKET_HEADER_NAME, SERVICE_TOKEN);
        request.addHeader(USER_TICKET_HEADER_NAME, USER_TOKEN);

        CheckedServiceTicket serviceTicket = mock(CheckedServiceTicket.class);
        when(serviceTicket.booleanValue()).thenReturn(true);
        when(tvmClient.checkServiceTicket(SERVICE_TOKEN)).thenReturn(serviceTicket);

        CheckedUserTicket userTicket = mock(CheckedUserTicket.class);
        when(userTicket.getStatus()).thenReturn(TicketStatus.EXPIRED);
        when(userTicket.booleanValue()).thenReturn(false);
        when(tvmClient.checkUserTicket(USER_TOKEN)).thenReturn(userTicket);

        UserAuthenticationHolder authenticationInput = new UserAuthenticationHolder(request);
        AuthenticationI18nException exception = assertThrows(AuthenticationI18nException.class,
                () -> tvmAuthenticationProvider.authenticate(authenticationInput));

        assertEquals(MsgWithArgs.of(AuthErrorMsg.USER_TICKET_STATUS, TicketStatus.EXPIRED), exception.getI18nMessage());

    }

    @Test
    void testHeaderUserIdAccess() {
        long userId = (long) (Math.random() * 1_000_000_000);

        MockHttpServletRequest request = new MockHttpServletRequest();
        request.addHeader(SERVICE_TICKET_HEADER_NAME, SERVICE_TOKEN);
        request.addHeader(USER_ID_HEADER, Long.toString(userId));
        UserAuthenticationHolder authenticationInput = new UserAuthenticationHolder(request);

        CheckedServiceTicket serviceTicket = mock(CheckedServiceTicket.class);
        when(serviceTicket.booleanValue()).thenReturn(true);
        when(serviceTicket.getSrc()).thenReturn(selfId);
        when(tvmClient.checkServiceTicket(SERVICE_TOKEN)).thenReturn(serviceTicket);

        Authentication authenticationOutput = tvmAuthenticationProvider.authenticate(authenticationInput);

        assertEquals(UserAuthenticationHolder.class, authenticationOutput.getClass());

        UserAuthentication authentication = (UserAuthentication) authenticationOutput.getDetails();

        assertEquals(AuthenticationMethod.AUTH_VIA_TVM_SERVICE_AND_HEADER, authentication.getAuthenticationMethod());
        assertEquals(userId, authentication.getUid());
        assertTrue(authentication.isSelfAuth());
    }

    @Test
    void testHeaderUserIdDenied() {
        long userId = (long) (Math.random() * 1_000_000_000);

        MockHttpServletRequest request = new MockHttpServletRequest();
        request.addHeader(SERVICE_TICKET_HEADER_NAME, SERVICE_TOKEN);
        request.addHeader(USER_ID_HEADER, Long.toString(userId));
        UserAuthenticationHolder authenticationInput = new UserAuthenticationHolder(request);

        CheckedServiceTicket serviceTicket = mock(CheckedServiceTicket.class);
        when(serviceTicket.booleanValue()).thenReturn(true);
        when(serviceTicket.getSrc()).thenReturn(selfId - 100);
        when(tvmClient.checkServiceTicket(SERVICE_TOKEN)).thenReturn(serviceTicket);

        assertThrows(AuthenticationI18nException.class,
                () -> tvmAuthenticationProvider.authenticate(authenticationInput));
    }
}
