package ru.yandex.partner.libs.auth.service;

import java.util.Set;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.env.Environment;
import org.springframework.mock.web.MockHttpServletRequest;

import ru.yandex.partner.core.CoreTest;
import ru.yandex.partner.core.entity.user.model.User;
import ru.yandex.partner.core.entity.user.service.UserService;
import ru.yandex.partner.libs.auth.exception.response.FakeLoginBadRequestException;
import ru.yandex.partner.libs.auth.exception.response.FakeLoginForbiddenException;
import ru.yandex.partner.libs.auth.model.FakeUserAuthentication;
import ru.yandex.partner.libs.auth.model.UserAuthentication;
import ru.yandex.partner.libs.auth.model.UserCredentials;
import ru.yandex.partner.libs.rbac.right.Right;
import ru.yandex.partner.libs.rbac.role.RoleSet;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertNotNull;
import static org.junit.jupiter.api.Assertions.assertThrows;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;
import static ru.yandex.partner.libs.auth.model.AuthenticationMethod.AUTH_VIA_COOKIES;
import static ru.yandex.partner.libs.rbac.constants.Constants.FAKE_LOGIN_RIGHT_NAME;
import static ru.yandex.partner.libs.rbac.constants.Constants.PRODUCTION;

@CoreTest
class FakeLoginServiceTest {
    UserAuthentication userAuthenticationRubiconProject2015;
    UserAuthentication userAuthenticationYNDXDEVELOPER;
    UserAuthentication userAuthenticationYNDXDEVELOPER2;

    @Autowired
    UserService userService;
    Environment mockEnv;
    FakeLoginService fakeLoginService;

    @BeforeEach
    public void initialize() {
        mockEnv = mock(Environment.class);
        fakeLoginService = new FakeLoginService(userService, mockEnv);

        final User rubiconProject2015User = mock(User.class);
        when(rubiconProject2015User.getRoles()).thenReturn(Set.of(RoleSet.SITE_PARTNER));
        userAuthenticationRubiconProject2015 = new UserAuthentication(
                AUTH_VIA_COOKIES,
                new UserCredentials(0L, "rubiconproject2015",
                        Set.of(RoleSet.SITE_PARTNER),
                        Set.of(new Right("safdasdf", Set.of(RoleSet.SITE_PARTNER))),
                        Set.of())
        );
        userAuthenticationRubiconProject2015.setInternalUser(rubiconProject2015User);

        userAuthenticationRubiconProject2015.setInternalUser(rubiconProject2015User);


        final User mockYNDXDEVELOPERUser = mock(User.class);
        when(mockYNDXDEVELOPERUser.getRoles()).thenReturn(Set.of(RoleSet.DEVELOPER));
        userAuthenticationYNDXDEVELOPER = new UserAuthentication(
                AUTH_VIA_COOKIES,
                new UserCredentials(0, "yndx-developer",
                        Set.of(RoleSet.DEVELOPER),
                        Set.of(new Right(FAKE_LOGIN_RIGHT_NAME, Set.of(RoleSet.DEVELOPER))),
                        Set.of())
        );

        userAuthenticationYNDXDEVELOPER2 = new UserAuthentication(
                AUTH_VIA_COOKIES,
                new UserCredentials(1, "yndx-developer",
                        Set.of(RoleSet.DEVELOPER),
                        Set.of(new Right(FAKE_LOGIN_RIGHT_NAME, Set.of(RoleSet.DEVELOPER))),
                        Set.of())
        );
    }

    @Test
    void tryFakeLoginThrows() {
        when(mockEnv.getActiveProfiles()).thenReturn(new String[]{"123"});
        assertThrows(NullPointerException.class, () -> fakeLoginService.tryFakeLogin(null,
                generateMockRequest("fakeLogin")));

        assertThrows(FakeLoginForbiddenException.class,
                () -> fakeLoginService.tryFakeLogin(userAuthenticationRubiconProject2015,
                        generateMockRequest("")));

        assertThrows(FakeLoginBadRequestException.class,
                () -> fakeLoginService.tryFakeLogin(userAuthenticationYNDXDEVELOPER,
                        generateMockRequest("non-existent-login")));

        when(mockEnv.getActiveProfiles()).thenReturn(new String[]{"123"});
        assertThrows(FakeLoginForbiddenException.class,
                () -> fakeLoginService.tryFakeLogin(userAuthenticationRubiconProject2015,
                        generateMockRequest("login")));

        when(mockEnv.getActiveProfiles()).thenReturn(new String[]{PRODUCTION});
        assertThrows(FakeLoginForbiddenException.class,
                () -> fakeLoginService.tryFakeLogin(userAuthenticationRubiconProject2015,
                        generateMockRequest("")));
    }


    @Test
    void tryFakeLoginSuccess() {
        when(mockEnv.getActiveProfiles()).thenReturn(new String[]{"123"});

        final FakeUserAuthentication actual =
                fakeLoginService.tryFakeLogin(userAuthenticationYNDXDEVELOPER2,
                        generateMockRequest("mocked-yan-partner"));
        assertEquals(AUTH_VIA_COOKIES, actual.getAuthenticationMethod());
        assertEquals(1009L, actual.getUid());
        assertNotNull(actual);
    }

    private HttpServletRequest generateMockRequest(String fakeLogin) {
        MockHttpServletRequest request = new MockHttpServletRequest();
        request.setCookies(new Cookie("fakelogin", fakeLogin));
        return request;
    }
}
