package ru.yandex.partner.libs.auth.service;

import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.servlet.http.HttpServletRequest;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

import ru.yandex.partner.core.entity.user.model.User;
import ru.yandex.partner.core.entity.user.service.UserAdfoxService;
import ru.yandex.partner.core.entity.user.service.UserService;
import ru.yandex.partner.core.role.Role;
import ru.yandex.partner.libs.auth.model.AuthenticationMethod;
import ru.yandex.partner.libs.auth.model.UserAuthentication;
import ru.yandex.partner.libs.auth.model.UserCredentials;
import ru.yandex.partner.libs.extservice.blackbox.BlackboxService;
import ru.yandex.partner.libs.rbac.right.Right;
import ru.yandex.partner.libs.rbac.role.RoleSet;
import ru.yandex.partner.libs.rbac.userrole.UserRoleService;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

public class UserDetailsServiceTest {

    private UserService userService;
    private UserAdfoxService userAdfoxService;
    private UserRoleService userRoleService;
    private FakeLoginService fakeLoginService;
    private BlackboxService blackboxService;


    private UserDetailsService userDetailsService;

    @BeforeEach
    public void prepare() {
        userService = mock(UserService.class);
        userAdfoxService = mock(UserAdfoxService.class);
        userRoleService = mock(UserRoleService.class);
        fakeLoginService = mock(FakeLoginService.class);
        blackboxService = mock(BlackboxService.class);

        userDetailsService = new UserDetailsService(
                userService,
                userAdfoxService,
                userRoleService,
                fakeLoginService,
                blackboxService
        );
    }

    @Test
    public void testAddDetails() {
        long id = 200;
        long uid = 274;
        String login = "usser";

        UserAuthentication input = new UserAuthentication(
                AuthenticationMethod.AUTH_VIA_API_KEYS_OAUTH, new UserCredentials(uid));

        Role role = mock(Role.class);
        Set<Role> roles = Set.of(role);

        User user = mock(User.class);
        when(user.getId()).thenReturn(id);
        when(user.getUid()).thenReturn(uid);
        when(user.getLogin()).thenReturn(login);
        when(user.getRoles()).thenReturn(roles);

        when(userService.findAll(any())).thenReturn(List.of(user));

        Right right = mock(Right.class);
        Set<Right> rights = Set.of(right);

        when(userRoleService.getUserRightsFromRoles(roles)).thenReturn(rights);

        UserAuthentication output = userDetailsService.addDetails(input, mock(HttpServletRequest.class));

        assertEquals(AuthenticationMethod.AUTH_VIA_API_KEYS_OAUTH, output.getAuthenticationMethod());
        assertEquals(uid, output.getUid());
        assertEquals(login, output.getLogin());
        assertEquals(rights, output.getUserCredentials().getRights());
        assertEquals(user, output.getInternalUser());

    }

    @Test
    public void testAddDetailsForSystemCronUser() {
        long uid = 0L;
        UserAuthentication input =
                new UserAuthentication(AuthenticationMethod.AUTH_VIA_TVM_SERVICE_AND_HEADER, new UserCredentials(uid));

        User user = new User();
        user.setUid(uid);
        user.setLogin("system-cron-test-user");

        when(
            userService.findAll(any())
        ).thenReturn(List.of(user));

        Set<Right> expectedRight = new HashSet<>(userRoleService.getUserRightsFromRoles(RoleSet.getRoles()));
        expectedRight.add(new Right(UserDetailsService.CAN_VIEW_ADFOX_FIELDS, Set.of()));

        UserAuthentication output = userDetailsService.addDetails(input, mock(HttpServletRequest.class));

        assertEquals(AuthenticationMethod.AUTH_VIA_TVM_SERVICE_AND_HEADER, output.getAuthenticationMethod());
        assertEquals(uid, output.getUid());
        assertEquals(user.getLogin(), output.getLogin());
        assertEquals(expectedRight, output.getUserCredentials().getRights());
        assertEquals(Set.of(), output.getRoles());
        assertEquals(Set.of(), output.getUserCredentials().getFeatures());
        assertEquals(user, output.getInternalUser());
    }

}
