package ru.yandex.partner.libs.authorization.decision;

import java.util.ArrayList;
import java.util.List;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;

import ru.yandex.partner.core.filter.CoreFilterNode;
import ru.yandex.partner.core.filter.operator.BinaryOperator;

public abstract class AuthorizationDecision<T> {
    private static final AuthorizationDecision<?> UNAPPLICABLE = new AuthorizationDecision<>() {
        @Override
        public <C> C match(
                Function<? super Boolean, ? extends C> unapplicable,
                Function<? super Boolean, ? extends C> permit,
                Function<? super CoreFilterNode<Object>, ? extends C> restrictedPermit,
                Function<? super String, ? extends C> deny
        ) {
            return unapplicable.apply(true);
        }

        @Override
        public String toString() {
            return "UNAPPLICABLE";
        }
    };

    private static final AuthorizationDecision<?> PERMIT = new AuthorizationDecision<>() {
        @Override
        public <C> C match(
                Function<? super Boolean, ? extends C> unapplicable,
                Function<? super Boolean, ? extends C> permit,
                Function<? super CoreFilterNode<Object>, ? extends C> restrictedPermit,
                Function<? super String, ? extends C> deny
        ) {
            return permit.apply(true);
        }

        @Override
        public String toString() {
            return "PERMIT";
        }
    };

    private AuthorizationDecision() {

    }

    public abstract <C> C match(
            Function<? super Boolean, ? extends C> unapplicable,
            Function<? super Boolean, ? extends C> permit,
            Function<? super CoreFilterNode<T>, ? extends C> restrictedPermit,
            Function<? super String, ? extends C> deny
    );

    public <C> C match(DecisionMatcher<T, C> matcher) {
        return match(
                matcher::unapplicable,
                matcher::permit,
                matcher::restrictedPermit,
                matcher::deny
        );
    }

    public static <T> AuthorizationDecision<T> unapplicable() {
        return (AuthorizationDecision<T>) UNAPPLICABLE;
    }

    public static <T> AuthorizationDecision<T> permit() {
        return (AuthorizationDecision<T>) PERMIT;
    }

    public static <T> AuthorizationDecision<T> restrictedPermit(CoreFilterNode<T> subset) {
        return new AuthorizationDecision<>() {
            @Override
            public <C> C match(
                    Function<? super Boolean, ? extends C> unapplicable,
                    Function<? super Boolean, ? extends C> permit,
                    Function<? super CoreFilterNode<T>, ? extends C> restrictedPermit,
                    Function<? super String, ? extends C> deny
            ) {
                return restrictedPermit.apply(subset);
            }
        };
    }

    public static <T> AuthorizationDecision<T> deny(String value) {
        return new AuthorizationDecision<>() {
            @Override
            public <C> C match(
                    Function<? super Boolean, ? extends C> unapplicable,
                    Function<? super Boolean, ? extends C> permit,
                    Function<? super CoreFilterNode<T>, ? extends C> restrictedPermit,
                    Function<? super String, ? extends C> deny
            ) {
                return deny.apply(value);
            }
        };
    }

    public boolean isUnapplicable() {
        return this == UNAPPLICABLE;
    }

    public boolean isPermitted() {
        return this == PERMIT;
    }

    public CoreFilterNode<T> restriction() {
        return this.match(
                unapplicable -> CoreFilterNode.alwaysFalse(),
                permit -> CoreFilterNode.neutral(),
                restrictedPermit -> restrictedPermit == null ? CoreFilterNode.neutral() : restrictedPermit,
                deny -> CoreFilterNode.alwaysFalse()
        );
    }

    public String denyReason() {
        return this.match(
                unapplicable -> null,
                permit -> null,
                restrictedPermit -> null,
                deny -> deny
        );
    }

    public void consume(
            Consumer<? super Boolean> unapplicableCase,
            Consumer<? super Boolean> permitCase,
            Consumer<? super CoreFilterNode<T>> restrictedPermitCase,
            Consumer<? super String> denyCase
    ) {
        this.match(
                unapplicable -> {
                    unapplicableCase.accept(unapplicable);
                    return null;
                },
                permit -> {
                    permitCase.accept(permit);
                    return null;
                },
                restrictedPermit -> {
                    restrictedPermitCase.accept(restrictedPermit);
                    return null;
                },
                deny -> {
                    denyCase.accept(deny);
                    return null;
                }
        );
    }

    @SafeVarargs
    public static <T> AuthorizationDecision<T> firstApplicable(AuthorizationDecision<T>... decisions) {
        return firstApplicable(List.of(decisions));
    }

    public static <T> AuthorizationDecision<T> firstApplicable(List<AuthorizationDecision<T>> decisions) {
        return decisions
                .stream()
                .filter(d -> !d.isUnapplicable())
                .findFirst()
                .orElseGet(AuthorizationDecision::unapplicable);
    }

    public static <T> AuthorizationDecision<T> onlyOneApplicable(List<AuthorizationDecision<T>> decisions) {
        ArrayList<AuthorizationDecision<T>> applicableDecisions =
                decisions
                        .stream()
                        .filter(d -> !d.isUnapplicable())
                        .collect(Collectors.toCollection(ArrayList::new));

        if (applicableDecisions.size() != 1) {
            throw new IllegalStateException("Wrong number of applicable decisions: " + decisions.size());
        }

        return applicableDecisions.get(0);
    }

    @SafeVarargs
    public static <T> AuthorizationDecision<T> denyOverrides(AuthorizationDecision<T>... decisions) {
        return denyOverrides(List.of(decisions));
    }

    private static <T> void filterDecisions(List<AuthorizationDecision<? super T>> decisions,
                                            List<AuthorizationDecision<? super T>> unapplicableDecisions,
                                            List<AuthorizationDecision<? super T>> permitDecisions,
                                            List<CoreFilterNode<? super T>> restrictions,
                                            List<AuthorizationDecision<? super T>> denyDecisions) {
        decisions.forEach(decision -> decision.match(
                unapplicable -> unapplicableDecisions.add(decision),
                permit -> permitDecisions.add(decision),
                restrictions::add,
                deny -> denyDecisions.add(decision))
        );
    }

    /**
     * Запрещает доступ если есть хотя бы один DENY decision.
     * Иначе достает ограничения, и reduce-ит фильтры через AND.
     * Если ограничений нет, то permit
     *
     * @param decisions
     * @param <T>
     * @return
     */
    public static <T> AuthorizationDecision<T> denyOverrides(List<AuthorizationDecision<? super T>> decisions) {
        List<AuthorizationDecision<? super T>> unapplicableDecisions = new ArrayList<>();
        List<AuthorizationDecision<? super T>> permitDecisions = new ArrayList<>();
        List<CoreFilterNode<? super T>> restrictions = new ArrayList<>();
        List<AuthorizationDecision<? super T>> denyDecisions = new ArrayList<>();

        filterDecisions(decisions, unapplicableDecisions, permitDecisions, restrictions, denyDecisions);

        if (unapplicableDecisions.size() == decisions.size()) {
            return AuthorizationDecision.unapplicable();
        }

        if (!denyDecisions.isEmpty()) {
            // TODO: multiple reasons
            return AuthorizationDecision.deny(denyDecisions.get(0).denyReason());
        }

        if (!restrictions.isEmpty()) {
            return AuthorizationDecision.restrictedPermit(new CoreFilterNode<>(BinaryOperator.AND, restrictions));
        }

        if (!permitDecisions.isEmpty()) {
            return AuthorizationDecision.permit();
        }

        throw new IllegalStateException("unreachable");
    }

    @SafeVarargs
    public static <T> AuthorizationDecision<T> permitOverrides(AuthorizationDecision<? super T>... decisions) {
        return AuthorizationDecision.<T>permitOverrides(List.of(decisions));
    }

    public static <T> AuthorizationDecision<T> permitOverrides(List<AuthorizationDecision<? super T>> decisions) {
        List<AuthorizationDecision<? super T>> unapplicableDecisions = new ArrayList<>();
        List<AuthorizationDecision<? super T>> permitDecisions = new ArrayList<>();
        List<CoreFilterNode<? super T>> restrictions = new ArrayList<>();
        List<AuthorizationDecision<? super T>> denyDecisions = new ArrayList<>();

        filterDecisions(decisions, unapplicableDecisions, permitDecisions, restrictions, denyDecisions);

        if (unapplicableDecisions.size() == decisions.size()) {
            return AuthorizationDecision.unapplicable();
        }

        if (!permitDecisions.isEmpty()) {
            return AuthorizationDecision.permit();
        }

        if (!restrictions.isEmpty()) {
            return AuthorizationDecision.restrictedPermit(new CoreFilterNode<>(BinaryOperator.OR, restrictions));
        }

        if (!denyDecisions.isEmpty()) {
            // TODO: multiple reasons
            return AuthorizationDecision.deny(denyDecisions.get(0).denyReason());
        }

        throw new IllegalStateException("unreachable");
    }
}
