package ru.yandex.partner.libs.bs.json

import org.json.JSONArray
import org.json.JSONObject
import org.skyscreamer.jsonassert.Customization
import org.skyscreamer.jsonassert.JSONCompareMode
import org.skyscreamer.jsonassert.JSONCompareResult
import org.skyscreamer.jsonassert.comparator.CustomComparator
import org.skyscreamer.jsonassert.comparator.JSONCompareUtil
import java.math.BigDecimal

open class BaseBkDataComparator(
    /**
     * Таким ключам допустимо появляться в actual, при условии, что они имеют "пустые" значения
     */
    private val allowedEmptyUnexpectedKeys: Set<String> = setOf(),
    /**
     * Таким ключам безусловно допустимо появляться в actual (BlockID)
     */
    private val allowedUnexpectedKeys: Set<String> = setOf(),
    /**
     * Таким ключам допустимо пропадать в actual, при условии, что они также имеют "пустые" значения в expected
     */
    private val allowedEmptyExpectedKeys: Set<String> = setOf(),
    /**
     * Таким ключам разрешено "пропадать":
     * 1. Под них нет места в proto и они ни на что не влияют
     */
    private val allowedMissing: Set<String> = setOf(),
    /**
     * По таким путям в префиксах меняются ключи currency, value - д.б. с большой буквы,
     * но в БД не всегда так
     */
    private val matchKeysPaths: Set<String> = setOf(),
    /**
     * Собственно маппинг ключей на ключи сериализации из proto, отличается только case-ом
     */
    private val matchKeys: Map<String, String> = mapOf(),
    mode: JSONCompareMode,
    vararg customizations: Customization
) : CustomComparator(mode, *customizations) {
    override fun checkJsonObjectKeysActualInExpected(
        prefix: String, expected: JSONObject, actual: JSONObject,
        result: JSONCompareResult
    ) {
        val actualKeys = JSONCompareUtil.getKeys(actual)
        for (key in actualKeys) {
            if (!expected.has(key)) {
                if (matchKeys.containsValue(key)) {
                    // probably not unexpected
                    continue
                }

                if (allowedUnexpectedKeys.contains(key)) {
                    continue
                }

                if (allowedEmptyUnexpectedKeys.contains(key)) {
                    val valueByKey = actual.opt(key)
                    if (isEmptyJsonValue(valueByKey)) {
                        continue
                    }
                }
                result.unexpected(prefix, key)
            }
        }
    }

    override fun checkJsonObjectKeysExpectedInActual(
        prefix: String,
        expected: JSONObject,
        actual: JSONObject,
        result: JSONCompareResult
    ) {
        val expectedKeys = JSONCompareUtil.getKeys(expected)
        for (key in expectedKeys) {
            val expectedValue = expected[key]
            val matchKey = if (matchKeysPaths.any { prefix.contains(it) } && matchKeys.containsKey(key))
                matchKeys[key] else key

            if (actual.has(matchKey)) {
                val actualValue = actual[matchKey]
                compareValues(JSONCompareUtil.qualify(prefix, key), expectedValue, actualValue, result)
            } else if (!allowedMissing.contains(key)
                && !(allowedEmptyExpectedKeys.contains(key) && isEmptyJsonValue(expectedValue))
            ) {
                result.missing(prefix, key)
            }
        }
    }

    /**
     * Эти числа сериализуются как строки в java из proto, но в БД сейчас могут встречаться и
     * как числа, и как строки
     * Булики сериализуются в джаве как 1 или 0,
     * но в БД могут встречаться ещё и как "1" и "0"
     */
    override fun compareValues(prefix: String?, expectedValue: Any?, actualValue: Any?, result: JSONCompareResult?) {
        if (expectedValue is Number && actualValue is String
            || expectedValue is String && actualValue is Number
        ) {
            return super.compareValues(
                prefix,
                BigDecimal(expectedValue.toString()).stripTrailingZeros(),
                BigDecimal(actualValue.toString()).stripTrailingZeros(),
                result
            )
        }
        super.compareValues(prefix, expectedValue, actualValue, result)
    }

    private fun isEmptyJsonValue(value: Any?) =
        value == null
            || value == JSONObject.NULL
            || value is JSONObject && value.length() == 0
            || value is JSONArray && value.length() == 0
            || value is String && value.length == 0
}
