package ru.yandex.partner.libs.bs.json;

import java.io.IOException;
import java.io.StringWriter;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeFormatterBuilder;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;

import NPartner.Page;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.json.JsonWriteFeature;
import com.fasterxml.jackson.core.util.JsonGeneratorDelegate;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.PropertyNamingStrategy;
import com.fasterxml.jackson.databind.node.JsonNodeFactory;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.google.protobuf.Message;
import com.hubspot.jackson.datatype.protobuf.ProtobufJacksonConfig;
import com.hubspot.jackson.datatype.protobuf.ProtobufModule;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static ru.yandex.partner.libs.bs.json.ReversibleConversion.combine;
import static ru.yandex.partner.libs.bs.json.ReversibleConversion.each;
import static ru.yandex.partner.libs.bs.json.ReversibleConversion.jsonifyKey;
import static ru.yandex.partner.libs.bs.json.ReversibleConversion.mapKey;
import static ru.yandex.partner.libs.bs.json.ReversibleConversion.objectToArrayWithId;
import static ru.yandex.partner.libs.bs.json.ReversibleConversion.onPath;
import static ru.yandex.partner.libs.bs.json.ReversibleConversion.removeKey;
import static ru.yandex.partner.libs.bs.json.ReversibleConversion.simply;

public class BkDataConverter {
    private static final Logger LOGGER = LoggerFactory.getLogger(BkDataConverter.class);

    private final ObjectMapper pbObjectMapper;
    private final ReversibleConversion<JsonNode> blockBkDataConversion;
    private final ReversibleConversion<JsonNode> pageBkDataConversion;

    public BkDataConverter() {
        pbObjectMapper = new ObjectMapper();

        pbObjectMapper.getSerializationConfig()
                .with(JsonWriteFeature.WRITE_NUMBERS_AS_STRINGS);

        pbObjectMapper.setPropertyNamingStrategy(new PropertyNamingStrategy.PropertyNamingStrategyBase() {
            @Override
            public String translate(String propertyName) {
                return propertyName;
            }
        });

        pbObjectMapper.registerModule(
                new ProtobufModule(ProtobufJacksonConfig.builder()
                        .acceptLiteralFieldnames(true)
                        .build()
                )
        );

        blockBkDataConversion = reversibleBlockBkDataConversion(pbObjectMapper);
        pageBkDataConversion = reversiblePageBkDataConversion(pbObjectMapper, blockBkDataConversion);
    }

    public String convertBlockBackAndForth(String bkData) throws IOException {
        var bkDataValue = convertBlockJsonToProto(bkData);

        return convertProtoToJson(bkDataValue.getMessage());
    }

    public String convertPageBackAndForth(String bkData) throws IOException {
        var bkDataValue = convertPageJsonToProto(bkData);

        return convertProtoToJson(bkDataValue.getMessage());
    }

    public String convertProtoToJson(Page.TPartnerPage.TBlock bkDataValue) throws IOException {
        JsonNode bkDataTree = convertProtoToJsonTree(bkDataValue);
        var jsonValue = bkDataTree.toString();
        LOGGER.debug("after backward (after serialization): {}", jsonValue);

        return jsonValue;
    }

    public JsonNode convertProtoToJsonTree(Page.TPartnerPage.TBlock bkDataValue) throws IOException {
        var writer = new StringWriter();
        // protobuf serializer uses generator directly and never uses other registered serializers,
        // so we need to modify generator itself to write booleans as ints
        var generator = perlCompatJsonGenerator(writer);
        pbObjectMapper.writeValue(generator, bkDataValue);
        var bkDataValueToJson = pbObjectMapper.readTree(
                writer.toString()
        );

        return blockBkDataConversion.backward(bkDataValueToJson);
    }

    public String convertProtoToJson(Page.TPartnerPage bkDataValue) throws IOException {
        var writer = new StringWriter();
        // protobuf serializer uses generator directly and never uses other registered serializers,
        // so we need to modify generator itself to write booleans as ints
        var generator = perlCompatJsonGenerator(writer);
        pbObjectMapper.writeValue(generator, bkDataValue);
        var bkDataValueToJson = pbObjectMapper.readTree(
                writer.toString()
        );

        var jsonValue = pageBkDataConversion.backward(bkDataValueToJson).toString();
        LOGGER.debug("after backward (after serialization): {}", jsonValue);

        return jsonValue;
    }

    @NotNull
    private JsonGeneratorDelegate perlCompatJsonGenerator(StringWriter writer) throws IOException {
        return new JsonGeneratorDelegate(pbObjectMapper.createGenerator(writer)) {
            @Override
            public void writeBoolean(boolean state) throws IOException {
                delegate.writeNumber(state ? 1 : 0);
            }

            @Override
            public void writeNumber(long v) throws IOException {
                writeString(String.valueOf(v));
            }

            @Override
            public void writeNumber(int v) throws IOException {
                writeString(String.valueOf(v));
            }
        };
    }

    public ParseResult<Page.TPartnerPage.TBlock> convertBlockJsonToProto(String bkData)
            throws JsonProcessingException {
        var bkDataJsonTree = (ObjectNode) pbObjectMapper.readTree(bkData);

        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("Input bkdata: {}", bkDataJsonTree.toString());
        }

        var errorsCollector = new HashMap<List<String>, String>();

        var parsedBlock = pbObjectMapper.copy()
                .addHandler(new BkDataDeserializationProblemHandler(errorsCollector))
                .convertValue(
                        blockBkDataConversion.forward(bkDataJsonTree),
                        Page.TPartnerPage.TBlock.class
                );

        return new ParseResult<>(parsedBlock, errorsCollector);
    }

    public ParseResult<Page.TPartnerPage> convertPageJsonToProto(String bkData) throws JsonProcessingException {
        var bkDataJsonTree = (ObjectNode) pbObjectMapper.readTree(bkData);

        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("Input bkdata: {}", bkDataJsonTree.toString());
        }

        var errorsCollector = new HashMap<List<String>, String>();

        var parsedPage = pbObjectMapper.copy()
                .addHandler(new BkDataDeserializationProblemHandler(errorsCollector))
                .convertValue(
                        pageBkDataConversion.forward(bkDataJsonTree),
                        Page.TPartnerPage.class
                );

        return new ParseResult<>(parsedPage, errorsCollector);
    }

    public static ReversibleConversion<JsonNode> reversiblePageBkDataConversion(
            ObjectMapper om,
            ReversibleConversion<JsonNode> blockConversion
    ) {
        var nf = om.getNodeFactory();

        ReversibleConversion<JsonNode> optionsConverter = simply(
                value -> {
                    var optionsArray = nf.arrayNode();
                    for (String optionWithValue : value.asText().split(";")) {
                        if (optionWithValue.isEmpty()) {
                            continue;
                        }
                        var kv = optionWithValue.split("=");
                        if (kv.length == 0 || kv[0].isEmpty()) {
                            continue;
                        }
                        var obj = nf.objectNode();
                        obj.set("Key", nf.textNode(kv[0]));
                        if (kv.length > 1) {
                            obj.set("Value", nf.textNode(kv[1]));
                        }
                        optionsArray.add(obj);
                    }
                    return optionsArray;
                },
                array -> {
                    if (!array.isArray()) {
                        return array;
                    }
                    var optsAsString = StreamSupport.stream(array.spliterator(), false)
                            .map(ObjectNode.class::cast)
                            .<String>map(option -> {
                                var key = option.get("Key");
                                var value = option.get("Value");

                                if (value.isNull()) {
                                    return key.asText();
                                } else {
                                    return key.asText() + "=" + value.asText();
                                }
                            })
                            .collect(Collectors.joining(";"));

                    return nf.textNode(optsAsString);
                }
        );
        return ReversibleConversion.combine(
                dateToLong(nf, "CreateDate"),
                mapKey("PIEditTime", "EditTime", true),
                dateToLong(nf, "EditTime"),
                mapKey("UpdateTimePI", "UpdateTime", true),
                // no such field in bssoap
                dateToLong(nf, "UpdateTime"),
                mapKey("isPi2", "IsPi2", true),
                mapKey("sad", "Sad", true),
                mapKey("excludeddomains", "ExcludedDomains", true),
                objectToArrayWithId(nf, "PICategoryIAB", "CategoryID"),
                objectToArrayWithId(nf, "RtbBlocks", "BlockID"),
                onPath("RtbBlocks", each(blockConversion)),
                objectToArrayWithId(nf, "DirectBlocks", "BlockID"),
                onPath("DirectBlocks", each(blockConversion)),
                objectToArrayWithId(nf, "Places", "PlaceID"),
                onPath("Places", each(onPath("StripeType", simply(
                        node -> nf.textNode(
                                Page.TPartnerPage.TPlace.EStripeType
                                        .forNumber(node.asInt()).getValueDescriptor().getName()
                        ),
                        node -> nf.numberNode(
                                Page.TPartnerPage.TPlace
                                        .EStripeType.valueOf(node.asText()).getNumber()
                        )
                )))),
                simply(
                        node -> {
                            if (node.has("DSPInfo") && node.path("DSPInfo").isEmpty()) {
                                ((ObjectNode) node).remove("DSPInfo");
                            }
                            return node;
                        },
                        Function.identity()
                ),
                onPath("Options", optionsConverter),
                onPath("RtbVideo", combine(
                        objectToArrayWithId(nf, "Contents", "ContentID"),
                        objectToArrayWithId(nf, "Categories", "CategoryID"),
                        objectToArrayWithId(nf, "VmapIDs", "VmapID")
                )),
                splitString(nf, "Mirrors", ","),
                splitString(nf, "DisabledFlags", ","),
                splitString(nf, "ExcludedDomains", ","),
                onPath("SSPPageToken", simply(
                        node -> node.isTextual() && node.asText().length() == 0 ? nf.arrayNode() : node,
                        Function.identity()
                )),
                onPath("State", simply(
                        node -> nf.textNode(
                                Page.TPartnerPage.EState.forNumber(node.asInt()).getValueDescriptor().getName()
                        ),
                        node -> nf.numberNode(
                                Page.TPartnerPage.EState.valueOf(node.asText()).getNumber()
                        )
                ))
        );
    }

    @NotNull
    private static ReversibleConversion<JsonNode> dateToLong(JsonNodeFactory nf, String dateKey) {
        var dateFormatter = new DateTimeFormatterBuilder()
                .append(DateTimeFormatter.ISO_LOCAL_DATE)
                .appendLiteral(' ')
                .append(DateTimeFormatter.ISO_LOCAL_TIME)
                .toFormatter();

        return simply(
                node -> {
                    var dateNode = node.get(dateKey);
                    if (dateNode == null || !dateNode.isTextual()) {
                        return node;
                    }

                    ((ObjectNode) node).replace(
                            dateKey,
                            nf.numberNode(
                                    LocalDateTime.parse(dateNode.asText(), dateFormatter)
                                            .toInstant(ZoneOffset.UTC)
                                            .toEpochMilli() / 1000
                            )
                    );
                    return node;
                },
                node -> {
                    var dateNode = node.get(dateKey);
                    if (dateNode == null || (!dateNode.isNumber() && !dateNode.isTextual())) {
                        return node;
                    }

                    ((ObjectNode) node).replace(
                            dateKey,
                            nf.textNode(
                                    dateFormatter.format(LocalDateTime.ofInstant(
                                            Instant.ofEpochMilli(dateNode.asLong() * 1000),
                                            ZoneOffset.UTC
                                    ))
                            )
                    );
                    return node;
                }
        );
    }

    @NotNull
    private static ReversibleConversion<JsonNode> splitString(JsonNodeFactory nf, String key, String separator) {
        return simply(
                node -> {
                    var itemsStr = node.get(key);
                    if (itemsStr == null || !itemsStr.isTextual()) {
                        return node;
                    }
                    var itemsArr = nf.arrayNode();
                    Arrays.stream(itemsStr.asText().split(separator))
                            .map(nf::textNode)
                            .forEachOrdered(itemsArr::add);
                    ((ObjectNode) node).replace(key, itemsArr);
                    return node;
                },
                node -> {
                    var itemsArr = node.get(key);
                    if (itemsArr == null || !itemsArr.isArray()) {
                        return node;
                    }
                    ((ObjectNode) node)
                            .replace(key,
                                    nf.textNode(StreamSupport.stream(itemsArr.spliterator(), false)
                                            .map(JsonNode::asText)
                                            .collect(Collectors.joining(separator)))
                            );
                    return node;
                }
        );
    }

    /**
     * Конвертер для подготовки json-дерева блока к
     * парсингу в proto-месседж при помощи jackson+ProtobufModule.
     * Также поддерживает обратную конвертацию сериализованного в json proto-месседжа в формат,
     * похожий на perl-пейлоад.
     * <p>
     * Большинство преобразований json проводятся прямо в исходном дереве,
     * поэтому порядок преобразований имеет значение.
     */
    public static ReversibleConversion<JsonNode> reversibleBlockBkDataConversion(ObjectMapper om) {
        var nf = om.getNodeFactory();

        // some kebab feature-tags to protobuf enum value
        ReversibleConversion<String> lowerKebabToScreamingSnake = simply(
                s -> s.replace('-', '_').toUpperCase(),
                s -> s.replace('_', '-').toLowerCase()
        );

        return combine(
                // BS SOAP и Caesar не знают таких полей на этом уровне
                removeKey("campaign_id"),
                removeKey("page_id"),
                removeKey("application_id"),
                removeKey("imageHeight"),
                // менеджерская настройка в internal_mobile_app
                removeKey("CustomOptions"),
                jsonifyKey(om, "CustomBlockData"),
                objectToArrayWithId(nf, "Geo", "GeoID"),
                objectToArrayWithId(nf, "PICategoryIAB", "CategoryID"),
                objectToArrayWithId(nf, "Brand", "BrandID"),
                objectToArrayWithId(nf, "Article", "ArticleID"),
                objectToArrayWithId(nf, "AdTypeSet", "AdType",
                        lowerKebabToScreamingSnake,
                        simply(
                                value -> nf.objectNode().set("Value", value),
                                object -> object.get("Value")
                        )
                ),
                jsonifyKey(om, "RtbDesign"),
                jsonifyKey(om, "Design"),
                objectToArrayWithId(nf, "AdType", "AdType",
                        lowerKebabToScreamingSnake,
                        ReversibleConversion.identity()
                ),
                onPath("Geo", each(
                        mapKey("value", "Value", true),
                        mapKey("currency", "Currency", true)
                )),
                onPath("Brand", each(
                        mapKey("value", "Value", true),
                        mapKey("currency", "Currency", true)
                )),
                onPath("AdType", each(
                        mapKey("value", "Value", true),
                        mapKey("currency", "Currency", true)
                )),
                onPath("Article", each(
                        mapKey("value", "Value", true),
                        mapKey("currency", "Currency", true)
                )),
                onPath("WrapperPromo", combine(
                        mapKey("begin", "Begin", true),
                        mapKey("promo", "Promo", true)
                )),
                onPath("WrapperAds", combine(
                        mapKey("begin", "Begin", true),
                        mapKey("end", "End", true)
                )),
                onPath("DSPInfo", each(mapKey("DSPPageImpOptions", "PageDspOptions"))),
                onPath("DSPSettings", onPath("DSPBindMode", simply(
                        node -> nf.textNode(
                                Page.TPartnerPage.TBlock.TDSPSettings.EDSPBindMode.valueOf(
                                        node.asText().toUpperCase()).name()
                        ),
                        node -> nf.textNode(
                                Page.TPartnerPage.TBlock.TDSPSettings.EDSPBindMode.valueOf(
                                        node.asText()).name().toLowerCase()
                        )
                )))
        );
    }

    public ReversibleConversion<JsonNode> getBlockBkDataConversion() {
        return blockBkDataConversion;
    }

    public ReversibleConversion<JsonNode> getPageBkDataConversion() {
        return pageBkDataConversion;
    }

    public ObjectMapper getPbObjectMapper() {
        return pbObjectMapper;
    }

    public static class ParseResult<M extends Message> {
        private final M message;
        private final Map<List<String>, String> errors;

        public ParseResult(M message, Map<List<String>, String> errors) {
            this.message = message;
            this.errors = errors;
        }

        public M getMessage() {
            return message;
        }

        public Map<List<String>, String> getErrors() {
            return errors;
        }
    }
}
